.check_pos_int <- local({function(x, name) {
  if (!is.numeric(x) || length(x) != 1L || !is.finite(x) ||
      x <= 0 || x != as.integer(x)) {
    stop(sprintf("`%s` must be a positive integer.", name), call. = FALSE)
  }
}}
)

.check_nonneg_int <- function(x, name) {
  if (!is.numeric(x) || length(x) != 1L || !is.finite(x) ||
      x < 0 || x != as.integer(x)) {
    stop(sprintf("`%s` must be a nonnegative integer.", name), call. = FALSE)
  }
}


.check_data.enroll <- function (data.enroll){
  if (missing(data.enroll) || is.null(data.enroll)) {
    stop("`data.enroll` must be provided.", call. = FALSE)
  }
  if (!is.data.frame(data.enroll)) {
    stop("`data.enroll` must be a data.frame.", call. = FALSE)
  }
  if (nrow(data.enroll) < 1L) {
    stop("`data.enroll` must have at least 1 row.", call. = FALSE)
  }

  .req_enroll <- c("interarrivaltime", "enrollstatus")
  .miss_enroll <- setdiff(.req_enroll, names(data.enroll))
  if (length(.miss_enroll) > 0L) {
    stop(sprintf("`data.enroll` is missing required column(s): %s",
                 paste0("`", .miss_enroll, "`", collapse = ", ")),
         call. = FALSE)
  }

  # interarrivaltime
  .t_enroll <- data.enroll[["interarrivaltime"]]
  if (!is.numeric(.t_enroll) || length(.t_enroll) < 1L) {
    stop("`data.enroll$interarrivaltime` must be a numeric vector with length >= 1.", call. = FALSE)
  }
  if (any(!is.finite(.t_enroll))) {
    stop("`data.enroll$interarrivaltime` must be finite (no NA/NaN/Inf).", call. = FALSE)
  }
  if (any(.t_enroll <= 0)) {
    stop("`data.enroll$interarrivaltime` must have all values > 0.", call. = FALSE)
  }

  # enrollstatus
  .st_enroll <- data.enroll[["enrollstatus"]]
  if (length(.st_enroll) != length(.t_enroll)) {
    stop("`data.enroll$enrollstatus` must have the same length as `interarrivaltime`.", call. = FALSE)
  }
  if (any(is.na(.st_enroll)) || !all(.st_enroll %in% c(0L, 1L, 0, 1))) {
    stop("`data.enroll$enrollstatus` must be coded as 0/1 (1 = enrolled, 0 = administratively censored).",
         call. = FALSE)
  }

  if (length(.t_enroll) != nrow(data.enroll)) {
    stop("`data.enroll$interarrivaltime` must have length nrow(data.enroll).", call. = FALSE)
  }
  if (length(.st_enroll) != nrow(data.enroll)) {
    stop("`data.enroll$enrollstatus` must have length nrow(data.enroll).", call. = FALSE)
  }
  invisible(NULL)

}



.check_data.eventcensor <- function(data.eventcensor, blinded = TRUE) {

  if (missing(data.eventcensor) || is.null(data.eventcensor)) {
    stop("`data.eventcensor` must be provided.", call. = FALSE)
  }
  if (!is.data.frame(data.eventcensor)) {
    stop("`data.eventcensor` must be a data.frame.", call. = FALSE)
  }
  if (nrow(data.eventcensor) < 1L) {
    stop("`data.eventcensor` must have at least 1 row.", call. = FALSE)
  }

  ## ---- required columns (always) ----
  req <- c("time", "eventstatus", "censorstatus")
  miss <- setdiff(req, names(data.eventcensor))
  if (length(miss) > 0L) {
    stop(sprintf("`data.eventcensor` is missing required column(s): %s",
                 paste0("`", miss, "`", collapse = ", ")),
         call. = FALSE)
  }

  ## ---- optional ID ----
  if ("No" %in% names(data.eventcensor)) {
    No <- data.eventcensor[["No"]]
    if (!is.numeric(No) || length(No) != nrow(data.eventcensor) ||
        any(!is.finite(No)) || any(No != as.integer(No)) || any(No <= 0)) {
      stop("`data.eventcensor$No` must be a positive integer vector.", call. = FALSE)
    }
  }

  ## ---- time ----
  tt <- data.eventcensor[["time"]]
  if (!is.numeric(tt) || length(tt) != nrow(data.eventcensor) ||
      any(!is.finite(tt)) || any(tt <= 0)) {
    stop("`data.eventcensor$time` must be finite and > 0.", call. = FALSE)
  }

  ## ---- eventstatus ----
  ev <- data.eventcensor[["eventstatus"]]
  if (length(ev) != nrow(data.eventcensor) ||
      any(is.na(ev)) || !all(ev %in% c(0L, 1L, 0, 1))) {
    stop("`eventstatus` must be coded as 0/1.", call. = FALSE)
  }

  ## ---- censorstatus ----
  cs <- data.eventcensor[["censorstatus"]]
  if (length(cs) != nrow(data.eventcensor) ||
      any(is.na(cs)) || !all(cs %in% c(0L, 1L, 0, 1))) {
    stop("`censorstatus` must be coded as 0/1.", call. = FALSE)
  }

  ## ---- logical consistency: random censoring implies no event ----
  ev_i <- as.integer(ev)
  cs_i <- as.integer(cs)
  if (any(cs_i == 1L & ev_i == 1L)) {
    stop("Invalid data: `censorstatus == 1` (random censoring) must imply `eventstatus == 0`.",
         call. = FALSE)
  }

  ## ---- treatment indicator ----
  if (!isTRUE(blinded)) {
    if (!("trt" %in% names(data.eventcensor))) {
      stop("When `blinded = FALSE`, `data.eventcensor` must contain column `trt`.",
           call. = FALSE)
    }
    trt <- data.eventcensor[["trt"]]
    if (length(trt) != nrow(data.eventcensor) ||
        any(is.na(trt)) || !all(trt %in% c(0L, 1L, 0, 1))) {
      stop("When `blinded = FALSE`, `trt` must be fully observed and coded as 0/1.",
           call. = FALSE)
    }
  }

  ## ---- covariates: everything else ----
  drop_cols <- c("No", "trt", req)
  cov_cols <- setdiff(names(data.eventcensor), drop_cols)

  if (length(cov_cols) == 0L) {
    return(NULL)
  }

  cov_df <- data.eventcensor[, cov_cols, drop = FALSE]

  if (!all(vapply(cov_df, is.numeric, logical(1)))) {
    bad <- names(cov_df)[!vapply(cov_df, is.numeric, logical(1))]
    stop(sprintf("Covariate columns must be numeric. Non-numeric: %s",
                 paste0("`", bad, "`", collapse = ", ")),
         call. = FALSE)
  }

  cov <- as.matrix(cov_df)
  if (any(!is.finite(cov))) {
    stop("Covariate values must be finite (no NA/NaN/Inf).", call. = FALSE)
  }

  cov
}



.check_seed_list3 <- function(seed) {
  if (is.null(seed)) seed <- list(NULL)

  if (is.numeric(seed) && length(seed) == 1L && is.finite(seed) && seed == as.integer(seed)) {
    seed <- list(as.integer(seed))
  }

  if (!is.list(seed)) {
    stop("`seed` must be NULL, a single integer, or a list (e.g., list(NULL) or list(1,2,3)).",
         call. = FALSE)
  }

  seed <- rep_len(seed, 3L)
  for (k in 1:3) {
    if (!is.null(seed[[k]])) {
      if (!is.numeric(seed[[k]]) || length(seed[[k]]) != 1L || !is.finite(seed[[k]]) ||
          seed[[k]] != as.integer(seed[[k]])) {
        stop(sprintf("`seed[[%d]]` must be NULL or a single finite integer.", k), call. = FALSE)
      }
      seed[[k]] <- as.integer(seed[[k]])
    }
  }
  seed
}

.check_control_list3 <- function(control) {
  if (is.null(control)) control <- list(list())

  # allow a single named list -> wrap
  if (is.list(control) && !is.null(names(control)) && !any(names(control) == "") &&
      !any(vapply(control, is.list, logical(1)))) {
    control <- list(control)
  }

  if (!is.list(control)) {
    stop("`control` must be a named list or a list of named lists.", call. = FALSE)
  }

  control <- rep_len(control, 3L)

  for (k in 1:3) {
    if (!is.list(control[[k]])) {
      stop(sprintf("`control[[%d]]` must be a list.", k), call. = FALSE)
    }
    if (length(control[[k]]) > 0L && (is.null(names(control[[k]])) || any(names(control[[k]]) == ""))) {
      warning(sprintf("`control[[%d]]` should be a named list.", k), call. = FALSE)
    }
  }
  control
}


trial_data_sim<-function (simdata, accrual, eventtarget = NULL)
{
  maxlpfollowup<-NULL
  trt <- status <- enrolltime <- eventtime <- NULL
  simdata$dropout <- ifelse(simdata$status == 1, "N", "Y")
  ncensor0 <- simdata %>% subset(trt == 0 & status == 0) %>%
    dplyr::pull(status)
  ncensor1 <- simdata %>% subset(trt == 1 & status == 0) %>%
    dplyr::pull(status)
  ncensor0 <- length(ncensor0)
  ncensor1 <- length(ncensor1)
  dr0 <- ncensor0/(simdata %>% subset(trt == 0) %>% nrow)
  dr1 <- ncensor1/(simdata %>% subset(trt == 1) %>% nrow)
  dr <- (ncensor0 + ncensor1)/nrow(simdata)
  Info <- data.frame(N_patient = nrow(simdata), N_event = sum(simdata$status ==
                                                                1), Dropout_rate = round(dr, 4), Dropout_rate_control = round(dr0,
                                                                                                                              4), Dropout_rate_treatment = round(dr1, 4), FirstEnrollTime = round(min(accrual),
                                                                                                                                                                                                  4), LastEnrollTime = 0, DataCutTime = 0, DataCutInfo = "")
  simdata$enrolltime <- accrual
  simdata <- simdata %>% dplyr::mutate(eventtime = enrolltime + time)
  eventt <- simdata %>% dplyr::filter(status == 1) %>% dplyr::select(eventtime) %>%
    dplyr::arrange(by_group = eventtime)
  eventt <- as.array(eventt$eventtime)
  lastpatient <- max(simdata$enrolltime)
  if (is.null(maxlpfollowup)) {
    lptime <- lastpatient + Inf
  }
  else {
    lptime <- lastpatient + maxlpfollowup
  }
  if (is.null(eventtarget)) {
    targettime <- Inf
  }
  else if (length(eventt) < eventtarget) {
    targettime <- Inf
  }
  else {
    targettime <- eventt[eventtarget]
  }
  datacut <- min(targettime, lptime)
  if (datacut < max(simdata$eventtime)) {
    simdata$status[simdata$eventtime > datacut] <- 0
    simdata$time[simdata$eventtime > datacut] <- datacut -
      simdata$enrolltime[simdata$eventtime > datacut]
    tmpdata <- simdata %>% dplyr::filter(time < 0)
    if (nrow(tmpdata) > 0) {
      simdata <- simdata %>% dplyr::filter(time >= 0)
      #cm <- paste("warning(", "\"", "Reached target event early or last patient followup time reached. Trial stopped.",
                  #"\"", ")")
      #eval(parse(text = cm))
      Info$N_patient <- nrow(simdata)
      Info$N_event <- table(simdata$status)[2]
      Info$DataCutTime <- datacut
      if (targettime <= lptime) {
        Info$DataCutInfo <- "target event"
      }
      else {
        Info$DataCutInfo <- "last patient max followup time"
      }
    }
    else {
      Info$N_patient <- nrow(simdata)
      Info$N_event <- table(simdata$status)[2]
      Info$DataCutTime <- datacut
      if (targettime <= lptime) {
        Info$DataCutInfo <- "target event"
      }
      else {
        Info$DataCutInfo <- "last patient max followup time"
      }
    }
    Info$LastEnrollTime <- (max(simdata$enrolltime))
    return(list(data = simdata, Info = Info))
  }
  else {
    Info$DataCutTime <- (max(simdata$eventtime))
    Info$DataCutInfo <- "last censor or event time"
    Info$LastEnrollTime <- (max(simdata$enrolltime))
    return(list(data = simdata, Info = Info))
  }
}


#' Solves for one unknown Weibull parameter (shape, baseline scale, or median
#' survival time) given the other two under a proportional hazards Weibull model.
#'
#' @description
#' Solves for one unknown Weibull parameter (shape, baseline scale, or median
#' survival time) given the other two under a proportional hazards Weibull model.
#' The Weibull distribution may be parameterized either by the baseline hazard
#' scale \eqn{\lambda_0} or by the baseline time scale \eqn{\sigma_0}, as specified
#' by \code{scaletype}.
#'
#' When no covariates are supplied, the function operates on the baseline Weibull
#' distribution. When covariates are provided, the model assumes a proportional
#' hazards structure and the supplied \code{median.event} is interpreted as the
#' marginal population median, defined implicitly by
#' \deqn{\mathbb{E}_Z\{S(m \mid Z)\} = 0.5.}
#'
#' With covariates, the conditional survival function is
#' \deqn{S(t \mid Z) = \exp\{-\lambda_0 \, t^\rho \exp(Z^\top \beta)\},}
#' where \eqn{\lambda_0 = \sigma_0^{-\rho}} when \code{scaletype = "time"}.
#' Here \eqn{Z} denotes the covariate vector; its components are generated
#' independently according to \code{cov_type} and \code{cov_dist}.
#' The expectation over covariates is approximated by Monte Carlo simulation using
#' \code{S} draws from the distributions specified by \code{cov_type} and
#' \code{cov_dist}. The unknown parameter is obtained by numerical root finding
#' via \code{\link[stats]{uniroot}}.
#'
#' @param shape Numeric scalar. Weibull shape parameter \eqn{\rho > 0}.
#'
#' @param scale Numeric scalar. Weibull baseline scale parameter. Its meaning depends on
#'   \code{scaletype}:
#'   \itemize{
#'     \item \code{scaletype = "time"}: time-scale \eqn{\sigma_0 > 0} so that
#'       \eqn{S_0(t) = \exp\{-(t/\sigma_0)^\rho\}} (same parameterization as
#'       \code{\link[stats]{dweibull}}).
#'     \item \code{scaletype = "hazard"}: hazard-scale \eqn{\lambda_0 > 0} so that
#'       \eqn{S_0(t) = \exp\{-\lambda_0 t^\rho\}} (same parameterization as
#'       \code{simsurv::simsurv}).
#'   }
#'
#' @param median.event Numeric scalar. Median survival time. If covariates are provided,
#'   this is the *marginal* (population) median \eqn{m} defined by
#'   \deqn{\mathbb{E}_Z\{S(m \mid Z)\} = 0.5,}
#'   where the expectation is with respect to the covariate distribution implied by
#'   \code{cov_type} and \code{cov_dist}.
#'
#' @param scaletype Character string specifying the Weibull baseline scale
#'   parameterization. Must be either \code{"time"} or \code{"hazard"}.
#'
#' @param cov_type Character vector specifying the distribution type for each covariate.
#'   Each element must be \code{"binary"} or \code{"continuous"}.
#'   If \code{NULL}, the model has no covariates.
#'
#' @param cov_dist Numeric vector of the same length as \code{cov_type}, giving the
#'   covariate distribution parameters used for Monte Carlo integration:
#'   \itemize{
#'     \item for \code{"binary"} covariates: success probability \eqn{p \in (0,1)};
#'     \item for \code{"continuous"} covariates: standard deviation \eqn{\sigma_Z \ge 0}
#'       for \eqn{Z \sim N(0, \sigma_Z^2)}.
#'   }
#'
#' @param beta Numeric vector of regression coefficients, same length as \code{cov_type}.
#'
#' @param S Integer. Monte Carlo sample size used to approximate the marginal survival
#'   when covariates are provided.
#'
#' @param seed Integer random seed for covariate simulation; if \code{NULL}, the RNG state is not reset.
#'
#' @param interval Numeric vector of length 2 giving the lower and upper
#'   bounds for the root-finding procedure used to solve for the unknown
#'   parameter (shape, scale, or median). The interval must bracket the
#'   true solution, i.e., the function values at the two endpoints must
#'   have opposite signs.
#'
#' @param tol Numeric scalar. Convergence tolerance passed to \code{\link[stats]{uniroot}}.
#'   Smaller values increase accuracy but may require more iterations.
#'
#' @param maxiter Integer. Maximum number of iterations for \code{\link[stats]{uniroot}}.
#'   If convergence is not reached, an error is thrown.
#'
#' @return
#' A numeric scalar giving the solved parameter. Exactly one of \code{shape},
#' \code{scale}, and \code{median.event} must be \code{NULL}; the function returns
#' the corresponding missing value.
#'
#' When covariates are supplied, the returned value is based on the marginal
#' population survival distribution obtained by Monte Carlo integration.
#'
#' @keywords internal
#' @noRd
solveparam_weibull<-function(shape=NULL,median.event=NULL,scale=NULL,
                             scaletype,cov_type=NULL, cov_dist=NULL, beta=NULL,
                             S=20000,seed=123,interval = c(1e-6,200),
                             tol = .Machine$double.eps^0.25, maxiter = 1000){
  # check parameters
  if (!is.null(cov_type)&&any(!cov_type %in% c("binary", "continuous"))) {
    stop("Error in solveparam: elements of cov_type must be either 'binary' or 'continuous'.",call.=FALSE)
  }
  if (!is.null(cov_type) && (is.null(cov_dist) || is.null(beta))) {
    stop("Error in solveparam: if cov_type is provided, cov_dist and beta must also be provided.",call.=FALSE)
  }
  if (!is.null(cov_type)&&!is.null(cov_dist)&&any((cov_type == "binary") & (cov_dist <= 0 | cov_dist >= 1))) {
    stop("Error in solveparam: for binary covariates, cov_dist must be strictly between 0 and 1.",call.=FALSE)
  }
  if (!is.null(cov_type)&&!is.null(cov_dist)&&any((cov_type == "continuous") & cov_dist < 0)) {
    stop("Error in solveparam: for continuous covariates, cov_dist must be greater than or equal to 0.",call.=FALSE)
  }
  if (!is.null(cov_type)) {
    if (length(cov_type) != length(cov_dist) || length(cov_type) != length(beta)) {
      stop("Invalid input in solveparam: cov_type, cov_dist, and beta must have the same length.",call.=FALSE)
    }
  }
  scaletype <- tryCatch(
    match.arg(scaletype, choices = c("hazard", "time")),
    error = function(e) {
      stop("Error in solveparam: invalid 'scaletype'. Must be one of: 'hazard' or 'time'.", call. = FALSE)
    }
  )
  if (!is.numeric(interval) || length(interval) != 2 || interval[1] <= 0 || interval[2] <= interval[1]) {
    stop("Error in solveparam: interval must be a numeric vector c(lower, upper) with 0 < lower < upper.", call. = FALSE)
  }
  if (any(!is.finite(interval))) stop("Error in solveparam: interval must be finite.", call. = FALSE)

  n_miss <- sum(is.null(shape), is.null(scale), is.null(median.event))
  if (n_miss != 1) {
    stop("Error in solveparam: exactly one of shape, scale, median.event must be NULL.",call.=FALSE)
  }
  if (!is.null(shape) && (!is.numeric(shape) || length(shape)!=1 || shape <= 0)) stop("Error in solveparam: shape must be a positive numeric scalar.",call.=FALSE)
  if (!is.null(scale) && (!is.numeric(scale) || length(scale)!=1 || scale <= 0)) stop("Error in solveparam: scale must be a positive numeric scalar.",call.=FALSE)
  if (!is.null(median.event) && (!is.numeric(median.event) || length(median.event)!=1 || median.event <= 0)) stop("Error in solveparam: median.event must be a positive numeric scalar.",call.=FALSE)
  if (!is.numeric(S) || length(S) != 1 || is.na(S) || S < 1 || S != as.integer(S)) {
    stop("Error in solveparam: S must be a positive integer.", call. = FALSE)
  }
  S <- as.integer(S)
  if (!is.numeric(tol) || length(tol) != 1 || is.na(tol) || tol <= 0) {
    stop("Error in solveparam: tol must be a positive numeric scalar.", call. = FALSE)
  }
  if (!is.numeric(maxiter) || length(maxiter) != 1 || is.na(maxiter) ||
      maxiter < 1 || maxiter != as.integer(maxiter)) {
    stop("Error in solveparam: maxiter must be a positive integer.", call. = FALSE)
  }
  maxiter <- as.integer(maxiter)


  # if there are no covariates
  if(is.null(cov_type)){
    if(is.null(scale)){
      scale=(log(2))^(-1/shape)*median.event
      if(scaletype=='time'){return(scale)}
      else{return(scale^(-shape))}
    }
    if(is.null(shape)){
      if(scaletype=='time'){shape = log(log(2))/log(median.event/scale)}
      if(scaletype=='hazard'){shape = log(log(2)/scale)/log(median.event)}
      return(shape)
    }
    if(is.null(median.event)){
      if(scaletype=='time'){return(scale*(log(2))^(1/shape))}
      if(scaletype=='hazard'){return((log(2)/scale)^(1/shape))}
    }
  }

  # simulate a large number of covariates
  if (!is.null(seed)) {
    if (!is.numeric(seed) || length(seed) != 1 || is.na(seed)) {
      stop("Error in solveparam: seed.solveparam must be NULL or a single numeric value.", call. = FALSE)
    }
    set.seed(as.integer(seed))
  }
  p<-length(cov_type)
  if (p < 1) stop("Error in solveparam: cov_type must have length >= 1.", call. = FALSE)
  cov <- matrix(NA_real_, nrow = S, ncol = p)
  if (!is.numeric(cov_dist) || !is.numeric(beta)) {
    stop("Error in solveparam: cov_dist and beta must be numeric.", call. = FALSE)
  }
  if (any(!is.finite(cov_dist)) || any(!is.finite(beta))) {
    stop("Error in solveparam: cov_dist and beta must be finite.", call. = FALSE)
  }
  for (j in seq_len(p)){
    if(cov_type[j]=='binary'){cov[,j]<-rbinom(S,1,cov_dist[j])}
    else {cov[,j]<-rnorm(S,0,cov_dist[j])}
  }
  lp<-drop(cov%*%beta)
  if (any(!is.finite(lp))) stop("Error in solveparam: linear predictor is not finite.", call. = FALSE)

  # solve for hazard scale
  if(is.null(scale)){
    f_scale<-function(lambda0){mean(exp(-lambda0*exp(lp)*median.event^shape))-0.5}
    lambda0<-uniroot(f_scale,interval=interval,tol=tol, maxiter=maxiter)$root # hazard scale
    if(scaletype=='hazard'){return(lambda0)}
    else{scale<-lambda0^(-1/shape);return(scale)}
  }
  # solve for shape
  if(is.null(shape)){
    if(scaletype=='hazard'){f_shape<-function(rho){mean(exp(-scale*exp(lp)*median.event^rho))-0.5}}
    else{f_shape<-function(rho){mean(exp(-scale^(-rho)*exp(lp)*median.event^rho))-0.5}}
    rho<-uniroot(f_shape,interval=interval,tol=tol,maxiter=maxiter)$root
    return(rho)
  }

  # solve for median.event
  if(scaletype=='time'){lambda0<-scale^(-shape)}
  if(scaletype=='hazard'){
    lambda0<-scale
  }
  f_med <- function(m) {mean(exp(-lambda0 * exp(lp) * m^shape)) - 0.5}
  median.event <- uniroot(f_med, interval = interval, tol = tol, maxiter = maxiter)$root
  return(median.event)
}



#' Generate survival times from a proportional hazards model
#' with a log-logistic baseline
#'
#' @description
#' Generates random survival times from a proportional hazards (PH) model
#' with a log-logistic baseline distribution.
#' The baseline survival function is
#' \deqn{S_0(t) = \{1 + (t / b)^a\}^{-1},}
#' where \eqn{a} is the shape parameter and \eqn{b} is the scale parameter.
#' Covariates enter the model through a linear predictor \eqn{\eta = x^\top\beta},
#' such that the cumulative hazard function is
#' \deqn{H(t \mid x) = \exp(\eta)\, \log\{1 + (t / b)^a\}.}
#' This implies \deqn{S(t \mid x) = \{1 + (t / b)^a\}^{-\exp(\eta)}.}
#' Survival times are generated using inverse cumulative hazard sampling.
#'
#' @details
#' Unlike the Weibull distribution, the log-logistic family is not closed under
#' proportional hazards. When \code{lp != 0}, the conditional distribution of
#' \eqn{T \mid x} is not log-logistic (even though the PH structure holds by construction).
#'
#' @param n Positive integer sample size.
#' @param shape Positive scalar log-logistic shape \eqn{a}. recycled to length \code{n}.
#' @param scale Positive scalar log-logistic scale \eqn{b}. recycled to length \code{n}.
#' @param lp Numeric linear predictor \eqn{\eta}; recycled to length \code{n}.
#'
#' @return Numeric vector of length \code{n}.
#'
#' @seealso
#' \code{\link[flexsurv]{rllogis}} for simulation under a log-logistic
#' accelerated failure time model.
#'
#' @references
#' Kleinbaum, D. G., & Klein, M. (2012).
#' \emph{Survival Analysis: A Self-Learning Text}.
#' Springer.
#'
#' @keywords internal
#' @noRd
rloglogistic_PH <- function(n, shape, scale, lp) {
  if (!is.numeric(n) || length(n) != 1 || is.na(n) || n < 1 || n != as.integer(n))
    stop("n must be a positive integer.", call. = FALSE)
  n <- as.integer(n)
  if (!is.numeric(shape) || anyNA(shape) || any(shape <= 0))
    stop("shape must be numeric and > 0.", call. = FALSE)
  if (!is.numeric(scale) || anyNA(scale) || any(scale <= 0))
    stop("scale must be numeric and > 0.", call. = FALSE)
  if (any(!is.finite(shape))) stop("shape must be finite.", call. = FALSE)
  if (any(!is.finite(scale))) stop("scale must be finite.", call. = FALSE)


  shape <- rep_len(shape, n)
  scale <- rep_len(scale, n)

  if (!is.numeric(lp)) stop("lp must be numeric.", call. = FALSE)
  lp <- rep_len(lp, n)
  if (any(!is.finite(lp))) stop("lp must be finite.", call. = FALSE)

  U <- runif(n)
  t <- scale*(exp(-log(U)/exp(lp))-1)^(1/shape)
  t
}




#' Solve log-logistic parameters by matching the marginal median (PH construction)
#'
#' @description
#' Solves for exactly one unknown log-logistic quantity: \code{shape} (\eqn{a}),
#' \code{scale} (\eqn{b}), or \code{median.event} (\eqn{m}), given the other two,
#' under a proportional hazards (PH) model built from a log-logistic baseline.
#'
#' The baseline survival function is
#' \deqn{S_0(t) = \{1 + (t/b)^a\}^{-1}.}
#' Under PH with linear predictor \eqn{\eta = Z^\top \beta}, the conditional survival is
#' \deqn{S(t \mid Z) = S_0(t)^{\exp(\eta)} = \{1 + (t/b)^a\}^{-\exp(Z^\top \beta)}.}
#' Here \eqn{Z} denotes the covariate vector whose components are generated
#' independently according to \code{cov_type} and \code{cov_dist}.
#'
#' When covariates are supplied, \code{median.event} is interpreted as the *marginal*
#' (population) median \eqn{m} defined implicitly by
#' \deqn{\mathbb{E}_Z\{S(m \mid Z)\} = 0.5,}
#' where the expectation is approximated by Monte Carlo using \code{S} simulated
#' covariate vectors from the distributions specified by \code{cov_type} and
#' \code{cov_dist}. The unknown parameter is obtained via root finding with
#' \code{\link[stats]{uniroot}}.
#'
#' @param shape Positive numeric scalar. Log-logistic shape parameter \eqn{a > 0}.
#'   Set to \code{NULL} to solve for \eqn{a}.
#' @param scale Positive numeric scalar. Log-logistic scale parameter \eqn{b > 0}.
#'   Set to \code{NULL} to solve for \eqn{b}.
#' @param median.event Positive numeric scalar. Marginal median survival time. With covariates,
#'   this is the marginal (population) median \eqn{m} solving
#'   \eqn{\mathbb{E}_Z\{S(m \mid Z)\} = 0.5}. Set to \code{NULL} to solve for \eqn{m}.
#'
#' @param cov_type Character vector specifying the distribution type for each covariate.
#'   Each element must be \code{"binary"} or \code{"continuous"}.
#'   If \code{NULL}, the model has no covariates.
#'
#' @param cov_dist Numeric vector of the same length as \code{cov_type}, giving the
#'   covariate distribution parameters used for Monte Carlo integration:
#'   \itemize{
#'     \item for \code{"binary"} covariates: success probability \eqn{p \in (0,1)};
#'     \item for \code{"continuous"} covariates: standard deviation \eqn{\sigma_Z \ge 0}
#'       for \eqn{Z \sim N(0, \sigma_Z^2)}.
#'   }
#'
#' @param beta Numeric vector of regression coefficients, same length as \code{cov_type}.
#'
#' @param S Integer. Monte Carlo sample size used to approximate the marginal survival
#'   when covariates are provided.
#'
#' @param seed Integer random seed for covariate simulation; if \code{NULL}, the RNG state is not reset.
#'
#' @param interval Numeric vector of length 2 giving the lower and upper
#'   bounds for the root-finding procedure used to solve for the unknown
#'   parameter (shape, scale, or median). The interval must bracket the
#'   true solution, i.e., the function values at the two endpoints must
#'   have opposite signs.
#'
#' @param tol Numeric scalar. Convergence tolerance passed to \code{\link[stats]{uniroot}}.
#'   Smaller values increase accuracy but may require more iterations.
#'
#' @param maxiter Integer. Maximum number of iterations for \code{\link[stats]{uniroot}}.
#'   If convergence is not reached, an error is thrown.
#'
#' @return
#' A numeric scalar giving the solved parameter. Exactly one of \code{shape},
#' \code{scale}, and \code{median.event} must be \code{NULL}; the function returns
#' the corresponding missing value.
#'
#' When covariates are supplied, the returned value is based on the marginal
#' population survival distribution obtained by Monte Carlo integration.
#'
#' @keywords internal
#' @noRd
solveparam_log<-function(shape=NULL,median.event=NULL,scale=NULL,
                         cov_type=NULL, cov_dist=NULL, beta=NULL,
                         S=20000,seed=123,interval = c(1e-7,200),
                         tol = .Machine$double.eps^0.25, maxiter = 1000){

  # interval
  if (!is.numeric(interval) || length(interval) != 2 || any(!is.finite(interval)) ||
      interval[1] <= 0 || interval[2] <= interval[1]) {
    stop("Error in solveparam: interval must be a finite numeric vector c(lower, upper) with 0 < lower < upper.", call. = FALSE)
  }

  # exactly one unknown
  n_miss <- sum(is.null(shape), is.null(scale), is.null(median.event))
  if (n_miss != 1) {
    stop("Error in solveparam: exactly one of shape, scale, median.event must be NULL.", call. = FALSE)
  }

  # scalar checks (when provided)
  if (!is.null(shape) && (!is.numeric(shape) || length(shape) != 1 || !is.finite(shape) || shape <= 0))
    stop("Error in solveparam: shape must be a finite positive numeric scalar.", call. = FALSE)

  if (!is.null(scale) && (!is.numeric(scale) || length(scale) != 1 || !is.finite(scale) || scale <= 0))
    stop("Error in solveparam: scale must be a finite positive numeric scalar.", call. = FALSE)

  if (!is.null(median.event) && (!is.numeric(median.event) || length(median.event) != 1 || !is.finite(median.event) || median.event <= 0))
    stop("Error in solveparam: median must be a finite positive numeric scalar.", call. = FALSE)

  if (!is.numeric(S) || length(S) != 1 || is.na(S) || S < 1 || S != as.integer(S))
    stop("Error in solveparam: S must be a positive integer.", call. = FALSE)
  S <- as.integer(S)

  if (!is.numeric(tol) || length(tol) != 1 || is.na(tol) || tol <= 0)
    stop("Error in solveparam: tol must be a positive numeric scalar.", call. = FALSE)
  if (!is.numeric(maxiter) || length(maxiter) != 1 || is.na(maxiter) ||
      maxiter < 1 || maxiter != as.integer(maxiter))
    stop("Error in solveparam: maxiter must be a positive integer.", call. = FALSE)
  maxiter <- as.integer(maxiter)


  # ---- EARLY EXIT: no covariates ----
  if (is.null(cov_type)) {
    if (is.null(shape)) {
      stop("Error in solveparam: with no covariates, the log-logistic shape cannot be identified because the baseline median equals the scale.", call. = FALSE)
    }
    if (is.null(scale)) return(median.event)      # because baseline median = scale
    # if median.event is NULL, baseline median equals scale
    return(scale)
  }


  # covariate checks
  if (is.null(cov_dist) || is.null(beta)) {
    stop("Error in solveparam: if cov_type is provided, cov_dist and beta must also be provided.", call. = FALSE)
  }
  if (!is.numeric(cov_dist) || !is.numeric(beta)) {
    stop("Error in solveparam: cov_dist and beta must be numeric.", call. = FALSE)
  }
  if (any(!is.finite(cov_dist)) || any(!is.finite(beta))) {
    stop("Error in solveparam: cov_dist and beta must be finite.", call. = FALSE)
  }
  if (!is.character(cov_type) || any(!cov_type %in% c("binary", "continuous"))) {
    stop("Error in solveparam: elements of cov_type must be either 'binary' or 'continuous'.", call. = FALSE)
  }
  if (length(cov_type) != length(cov_dist) || length(cov_type) != length(beta)) {
    stop("Error in solveparam: cov_type, cov_dist, and beta must have the same length.", call. = FALSE)
  }
  if (any((cov_type == "binary") & (cov_dist <= 0 | cov_dist >= 1))) {
    stop("Error in solveparam: for binary covariates, cov_dist must be strictly between 0 and 1.", call. = FALSE)
  }
  if (any((cov_type == "continuous") & (cov_dist < 0))) {
    stop("Error in solveparam: for continuous covariates, cov_dist must be >= 0.", call. = FALSE)
  }

  # simulate covariates
  if (!is.null(seed)) {
    if (!is.numeric(seed) || length(seed) != 1 || is.na(seed))
      stop("Error in solveparam: seed.solveparam must be NULL or a single numeric value.", call. = FALSE)
    set.seed(as.integer(seed))
  }

  p <- length(cov_type)
  if (p < 1) stop("Error in solveparam: cov_type must have length >= 1.", call. = FALSE)
  cov <- matrix(NA_real_, nrow = S, ncol = p)
  for (j in seq_len(p)) {
    if (cov_type[j] == "binary") {cov[, j] <- rbinom(S, 1, cov_dist[j]) }
    else {cov[, j] <- rnorm(S, 0, cov_dist[j])}
  }
  lp <- drop(cov %*% beta)
  if (any(!is.finite(lp))) stop("Error in solveparam: linear predictor is not finite.", call. = FALSE)
  elp <- exp(lp)
  if (any(!is.finite(elp))) stop("Error in solveparam: exp(lp) overflowed; check beta/covariate scales.", call. = FALSE)

  # solve for scale
  if(is.null(scale)){
    f_scale<-function(alpha){mean((1+(median.event/alpha)^shape)^(-exp(lp)))-0.5}
    return(uniroot(f_scale,interval=interval,tol=tol, maxiter=maxiter)$root)
  }
  if(is.null(shape)){
    f_shape<-function(kappa){mean((1+(median.event/scale)^kappa)^(-exp(lp)))-0.5}
    return(uniroot(f_shape,interval=interval,tol=tol, maxiter=maxiter)$root)
  }
  if(is.null(median.event)){
    f_med<-function(m){mean((1+(m/scale)^shape)^(-exp(lp)))-0.5}
    return(uniroot(f_med,interval=interval,tol=tol, maxiter=maxiter)$root)
  }
}



ext_to_draws <- function(ext, pars) {
  stopifnot(is.array(ext), length(dim(ext)) == 3)

  df <- reshape2::melt(
    ext,
    varnames = c("iteration", "chain", "parameter"),
    value.name = "value"
  ) |>
    tibble::as_tibble()

  # robust parsing: works for 1, "1", "chain:1", "iter:1", etc.
  iter_id  <- readr::parse_number(as.character(df$iteration))
  chain_id <- readr::parse_number(as.character(df$chain))

  if (anyNA(iter_id) || anyNA(chain_id)) {
    stop("Could not parse iteration/chain indices from melted array. Check dimnames of `ext`.", call. = FALSE)
  }

  n_iter <- max(iter_id)

  df <- df |>
    dplyr::mutate(
      iteration = as.integer(iter_id),
      chain     = as.integer(chain_id),
      draw      = (chain - 1L) * n_iter + iteration
    )

  # keep only requested pars (including vector elements like beta[1])
  wanted <- paste0("^(", paste(pars, collapse = "|"), ")(\\[|$)")
  df <- df |> dplyr::filter(grepl(wanted, parameter))

  # (draw, parameter) should be unique; if not, something is wrong with draw id.
  dup <- df |>
    dplyr::count(draw, parameter) |>
    dplyr::filter(n > 1L)
  if (nrow(dup) > 0L) {
    stop("Duplicate (draw, parameter) detected: draw indexing is not unique. This usually means chain/iteration parsing failed.",
         call. = FALSE)
  }

  wide <- df |>
    dplyr::select(draw, parameter, value) |>
    tidyr::pivot_wider(
      names_from = parameter,
      values_from = value
    ) |>
    dplyr::arrange(draw)

  out <- list()

  for (p in pars) {
    cols <- grep(paste0("^", p, "(\\[|$)"), names(wide), value = TRUE)
    if (!length(cols)) {
      warning(sprintf("Parameter '%s' not found in ext.", p), call. = FALSE)
      next
    }

    if (length(cols) == 1) {
      out[[p]] <- wide[[cols]]            # scalar -> vector
    } else {
      cols <- cols[order(readr::parse_number(cols))]
      out[[p]] <- as.matrix(wide[, cols, drop = FALSE])  # vector -> matrix
    }
  }

  out
}



#' @keywords internal
utils::globalVariables(c(
  "chain", "iteration", "parameter", "draw", "n", "value"
))


ext_to_draws <- function(ext, pars) {
  stopifnot(is.array(ext), length(dim(ext)) == 3)

  df <- reshape2::melt(
    ext,
    varnames = c("iteration", "chain", "parameter"),
    value.name = "value"
  ) |>
    tibble::as_tibble()

  # robust parsing: works for 1, "1", "chain:1", "iter:1", etc.
  iter_id  <- readr::parse_number(as.character(df$iteration))
  chain_id <- readr::parse_number(as.character(df$chain))

  if (anyNA(iter_id) || anyNA(chain_id)) {
    stop("Could not parse iteration/chain indices from melted array. Check dimnames of `ext`.", call. = FALSE)
  }

  n_iter <- max(iter_id)

  df <- df |>
    dplyr::mutate(
      iteration = as.integer(iter_id),
      chain     = as.integer(chain_id),
      draw      = (chain - 1L) * n_iter + iteration
    )

  # keep only requested pars (including vector elements like beta[1])
  wanted <- paste0("^(", paste(pars, collapse = "|"), ")(\\[|$)")
  df <- df |> dplyr::filter(grepl(wanted, parameter))

  # (draw, parameter) should be unique; if not, something is wrong with draw id.
  dup <- df |>
    dplyr::count(draw, parameter) |>
    dplyr::filter(n > 1L)
  if (nrow(dup) > 0L) {
    stop("Duplicate (draw, parameter) detected - draw indexing is not unique. This usually means chain/iteration parsing failed.",
         call. = FALSE)
  }

  wide <- df |>
    dplyr::select(draw, parameter, value) |>
    tidyr::pivot_wider(
      names_from = parameter,
      values_from = value
    ) |>
    dplyr::arrange(draw)

  out <- list()

  for (p in pars) {
    cols <- grep(paste0("^", p, "(\\[|$)"), names(wide), value = TRUE)
    if (!length(cols)) {
      warning(sprintf("Parameter '%s' not found in ext.", p), call. = FALSE)
      next
    }

    if (length(cols) == 1) {
      out[[p]] <- wide[[cols]]            # scalar -> vector
    } else {
      cols <- cols[order(readr::parse_number(cols))]
      out[[p]] <- as.matrix(wide[, cols, drop = FALSE])  # vector -> matrix
    }
  }

  out
}
