% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/findGrowthStarts.R
\name{findGrowthStarts}
\alias{findGrowthStarts}
\title{Find reasonable starting values for common fish growth functions.}
\usage{
findGrowthStarts(
  formula,
  data,
  type = c("von Bertalanffy", "Gompertz", "logistic", "Richards", "Schnute",
    "Schnute-Richards"),
  param = 1,
  pname = NULL,
  case = NULL,
  constvals = NULL,
  fixed = NULL,
  plot = FALSE
)
}
\arguments{
\item{formula}{A formula of the form \code{length~age} for length-at-age models or \code{deltaL~deltat+lengthM} for tag-recapture models. \code{length} and \code{age} generically represent the observed length and annual age, and \code{deltaL}, \code{deltat}, and \code{lengthM} generically represent the observed change in length, observed change in time, and observed length at marking.}

\item{data}{A data frame that contains the variables in \code{formula}.}

\item{type}{A single string (i.e., one of \dQuote{von Bertalanffy}, \dQuote{Gompertz}, \dQuote{logistic}, \dQuote{Richards}, \dQuote{Schnute}, \dQuote{Schnute-Richards}) that indicates the type of growth function to show.}

\item{param}{A single numeric that indicates the specific parameterization of the growth function. Will be ignored if \code{pname} is non-\code{NULL}. See details.}

\item{pname}{A single character that indicates the specific parameterization of the growth function. If \code{NULL} then \code{param} will be used. See details.}

\item{case}{A numeric that indicates the specific case of the Schnute function to use.}

\item{constvals}{A NAMED numeric vector of constant values (either lengths or ages) to be used in some of the von Bertalanffy parameterizations. See details.}

\item{fixed}{A NAMED numeric vector that contains user-defined (i.e., fixed rather than automatically generated) starting values for one or more parameters. See details.}

\item{plot}{A logical that indicates whether a plot of the data with the superimposed model fit at the starting values should be created. This plot is for diagnostic purposes and, thus, cannot be modified in this function.}
}
\value{
A named vector that contains reasonable starting values. Note that the parameters will be listed with the same names in the same order as listed in \code{\link{makeGrowthFun}}.
}
\description{
Finds reasonable starting values for the parameters in a specific parameterization of common growth functions (von Bertalanffy, Gompertz, logistic, Richards, Schnute, and Schnute-Richards).
}
\details{
This function attempts to find reasonable starting values for a variety of parameterizations of common functions used to model fish growth (von Bertalanffy, Gompertz, logistic, Richards, Schnute, and Schnute-Richards). The starting values tend to work well in \code{nls} and related non-linear modeling functions, but there is no guarantee that they are the \sQuote{best} starting values (especially if the model is not appropriate for the data). One should perform sensitivity analyses to determine the impact of different starting values on the final model results.

In some instances it may be beneficial to fix one or more of the starting values to a user-defined choice. This can be done with \code{fixed} as shown in the examples. Note that starting values for other parameters that depend on the value of the fixed parameter may also be affected. For example, a starting value for \eqn{t_0} in the "Typical" von Bertalanffy function depends on values of \eqn{L_\infty} and \eqn{K}. Thus, if, for example, \eqn{K} is fixed by the user then the starting value for \eqn{t_0} will also be affected as it will used the fixed rather than the automatically derived value for \eqn{K}.

It is good practice for two reasons to use \code{plot=TRUE} to superimpose the growth function evaluated at the starting values over a scatterplot of the observed lengths versus ages. First, this will give the user a feel for how well the growth function fits the data given the starting values. If the "model line" does not represent the data well then the starting values are likely poor and the non-linear model may not converge. Second, the user may iteratively supply values for the parameters in \code{fixed} with \code{plot=TRUE} to "guess" at useful starting values. This is demonstrated in the examples.

See this \href{https://fishr-core-team.github.io/FSA/articles/Fitting_Growth_Functions.html}{article} for complete examples of "fitting" growth models with FSA.
}
\note{
Derivation of the starting values is detailed in \href{https://fishr-core-team.github.io/FSA/articles/Growth_Starting_Values.html}{this article}. Further note that starting values have not yet been coded for every parameterization of the growth functions available in \code{FSA}. In those instances, you will need to derive starting values by other means.
}
\section{IFAR Chapter}{
 12-Individual Growth.
}

\examples{
# These examples use the hypothetical length-at-age (annual) data in GrowthData1

#===== Example starting values for 1st parameterization of each type
( svonb1 <- findGrowthStarts(tlV~age,data=GrowthData1,type="von Bertalanffy") )
( sgomp1 <- findGrowthStarts(tlG~age,data=GrowthData1,type="Gompertz") )
( slogi1 <- findGrowthStarts(tlL~age,data=GrowthData1,type="logistic") )
( srich1 <- findGrowthStarts(tlR~age,data=GrowthData1,type="Richards") )

#====== Example starting values at other parameterizations
( svonb4 <- findGrowthStarts(tlV~age,data=GrowthData1,type="von Bertalanffy",param=4) )
( sgomp2 <- findGrowthStarts(tlG~age,data=GrowthData1,type="Gompertz",param=2) )
( slogi3 <- findGrowthStarts(tlL~age,data=GrowthData1,type="logistic",param=3) )
( srich3 <- findGrowthStarts(tlR~age,data=GrowthData1,type="Richards",param=3) )

#' #====== Example using pname instead of param
( svonb4 <- findGrowthStarts(tlV~age,data=GrowthData1,type="von Bertalanffy",pname="Mooij") )
( sgomp2 <- findGrowthStarts(tlG~age,data=GrowthData1,type="Gompertz",pname="Ricker1") )
( slogi3 <- findGrowthStarts(tlL~age,data=GrowthData1,type="logistic",pname="Campana-Jones2") )
( srich3 <- findGrowthStarts(tlR~age,data=GrowthData1,type="Richards",pname="Tjorve7") )

#====== Some vonB parameterizations require constant values in constvals=
( svonb8 <- findGrowthStarts(tlV~age,data=GrowthData1,type="von Bertalanffy",
                             pname="Francis",constvals=c(t1=2,t3=11)) )

#====== Demonstrate use of fixed= with 2nd (Original) param of von B as e.g.
( svonb2 <- findGrowthStarts(tlV~age,data=GrowthData1,param=2) )
( svonb2 <- findGrowthStarts(tlV~age,data=GrowthData1,param=2,fixed=c(Linf=500)) )
( svonb2 <- findGrowthStarts(tlV~age,data=GrowthData1,param=2,fixed=c(Linf=500,K=0.25)) )

#===== Starting values with diagnostic plot
( sgomp3 <- findGrowthStarts(tlG~age,data=GrowthData1,type="Gompertz",param=3,plot=TRUE) )

#===== Iteratively guess at starting values (stop when the model seems to "fit")
findGrowthStarts(tlV~age,data=GrowthData1,plot=TRUE,fixed=c(Linf=600,K=0.5,t0=0))    #att 1
findGrowthStarts(tlV~age,data=GrowthData1,plot=TRUE,fixed=c(Linf=450,K=0.5,t0=0))    #att 2
findGrowthStarts(tlV~age,data=GrowthData1,plot=TRUE,fixed=c(Linf=450,K=0.3,t0=0))    #att 3
findGrowthStarts(tlV~age,data=GrowthData1,plot=TRUE,fixed=c(Linf=450,K=0.3,t0=-0.5)) #looks OK, stop

#===== Plot at starting and final values
#----- creating growth function corresponding to first param of von B
vonb1 <- makeGrowthFun(type="von Bertalanffy")
#----- plot data
plot(tlV~age,data=GrowthData1,pch=19,col=col2rgbt("black",0.2))
#----- plot von b growth function at starting values (svonb1 from above)
curve(vonb1(x,Linf=svonb1),col="blue",lwd=5,add=TRUE)
#----- fit growth function to data
rvonb1 <- nls(tlV~vonb1(age,Linf,K,t0),data=GrowthData1,start=svonb1)
cvonb1 <- coef(rvonb1)
#----- plot growth function at final values ... starting values were very good!
curve(vonb1(x,Linf=cvonb1),col="red",lwd=2,add=TRUE)

#===== Example for tag-recapture data (in GrowthData3)
#----- Fabens model
findGrowthStarts(deltaL~deltat+tlM,data=GrowthData3,pname="Fabens")
#----- Francis model
findGrowthStarts(deltaL~deltat+tlM,data=GrowthData3,pname="Francis2",
                 constvals=c(L1=150,L2=400))

}
\seealso{
See \code{\link{makeGrowthFun}} to make functions that use these starting values and \code{\link{showGrowthFun}} to display the equations used in \pkg{FSA}. See \code{\link{nlsTracePlot}} for help troubleshooting nonlinear models that don't converge.
}
\author{
Derek H. Ogle, \email{DerekOgle51@gmail.com}
}
\keyword{manip}
