% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/KFPCA.R
\name{KFPCA}
\alias{KFPCA}
\title{Kendall Functional Principal Component Analysis (KFPCA) for sparse design}
\usage{
KFPCA(
  Lt,
  Ly,
  interval,
  dataType = "Sparse",
  nK,
  kern = "epan",
  bw,
  kernK = "epan",
  bwK = "GCV",
  kernmean = "epan",
  bwmean = "GCV",
  nRegGrid,
  fdParobj,
  more = TRUE
)
}
\arguments{
\item{Lt}{A \code{list} of \emph{n} vectors, where \emph{n} is the sample size. Each entry contains the observation time in ascending order for each subject.}

\item{Ly}{A \code{list} of \emph{n} vectors, where \emph{n} is the sample size. Each entry contains the measurements of each subject at the observation time correspond to \code{Lt}.}

\item{interval}{A \code{vector} of length two denoting the supporting interval.}

\item{dataType}{A \code{character} denoting the data type; 'Sparse'-default, 'Dense'.}

\item{nK}{An integer denoting the number of FPCs.}

\item{kern}{A \code{character} denoting the kernel type for the Nadaraya-Watson estimators; 'epan'(Epanechnikov)-default, 'unif'(Uniform), 'quar'(Quartic), 'gauss'(Gaussian).}

\item{bw}{A scalar denoting the bandwidth for the Nadaraya-Watson estimators.}

\item{kernK}{A \code{character} denoting the kernel type for the estimation of the Kendall's tau function; 'epan'(Epanechnikov)-default, 'unif'(Uniform), 'quar'(Quartic), 'gauss'(Gaussian).}

\item{bwK}{The bandwidth for the estimation of the Kendall's tau function. If \code{is.numeric(bwK) == T}, \code{bwK} is exactly the bandwidth. If \code{bwK == "GCV"}, the bandwidth is chosen by GCV. (default: "GCV")}

\item{kernmean}{A \code{character} denoting the kernel type for the estimation of the mean function; 'epan'(Epanechnikov)-default, 'unif'(Uniform), 'quar'(Quartic), 'gauss'(Gaussian).}

\item{bwmean}{The bandwidth for the estimation of the mean function. If \code{is.numeric(bwmean) == T}, \code{bwmean} is exactly the bandwidth. If \code{bwmean == "GCV"}, the bandwidth is chosen by GCV. (default: "GCV")}

\item{nRegGrid}{An integer denoting the number of equally spaced time points in the supporting interval. The eigenfunctions and mean function are estimated at these equally spaced time points.}

\item{fdParobj}{A functional parameter object for the smoothing of the eigenfunctions. For more detail, see \code{\link[fda]{smooth.basis}}.}

\item{more}{Logical; If \code{FALSE}, estimates of FPC scores and predictions of trajectories are not returned.}
}
\value{
A \code{list} containing the following components:
\item{ObsGrid}{A \code{vector} containing all observation time points in ascending order.}
\item{RegGrid}{A \code{vector} of the equally spaced time points in the support interval.}
\item{bwmean}{A scalar denoting the bandwidth for the mean function estimate.}
\item{kernmean}{A \code{character} denoting the kernel type for the estimation of the mean function}
\item{bwK}{A scalar denoting the bandwidth for the Kendall's tau function estimate.}
\item{kernK}{A \code{character} denoting the kernel type for the estimation of the Kendall's tau function}
\item{mean}{A \code{vector} of length \code{nRegGrid} denoting the mean function estimate.}
\item{KendFun}{A \code{nRegGrid} by \code{nRegGrid} \code{matrix} denoting the Kendall's tau function estimate.}
\item{FPC_dis}{A \code{nRegGrid} by \code{nK} \code{matrix} containing the eigenfunction estimates at \code{RegGrid}.}
\item{FPC_smooth}{A functional data object for the eigenfunction estimates.}
\item{score}{A \emph{n} by \code{nK} \code{matrix} containing the estimates of the FPC scores, where \emph{n} is the sample size. The results are returned when \code{more = TRUE}.}
\item{X_fd}{A functional data object for the prediction of trajectories. The results are returned when \code{more = TRUE}.}
\item{Xest_ind}{A \code{list} containing the prediction of each trajectory at their own observation time points. The results are returned when \code{more = TRUE}.}
\item{Lt}{The input 'Lt'.}
\item{Ly}{The input 'Ly'.}
\item{CompTime}{A scalar denoting the computation time.}
}
\description{
KFPCA for non-Gaussian functional data with sparse design or longitudinal data.
}
\examples{
# Generate data
n <- 100
interval <- c(0, 10)
lambda_1 <- 9 #the first eigenvalue
lambda_2 <- 1.5 #the second eigenvalue
eigfun <- list()
eigfun[[1]] <- function(x){cos(pi * x/10)/sqrt(5)}
eigfun[[2]] <- function(x){sin(pi * x/10)/sqrt(5)}
score <- cbind(rnorm(n, 0, sqrt(lambda_1)), rnorm(n, 0, sqrt(lambda_2)))
DataNew <- GenDataKL(n, interval = interval, sparse = 6:8, regular = FALSE,
                     meanfun = function(x){0}, score = score,
                     eigfun = eigfun, sd = sqrt(0.1))
basis <- fda::create.bspline.basis(interval, nbasis = 13, norder = 4,
                              breaks = seq(0, 10, length.out = 11))
# KFPCA
Klist <- KFPCA(DataNew$Lt, DataNew$Ly, interval, nK = 2, bw = 1,
               nRegGrid = 51, fdParobj = basis)
plot(Klist$FPC_smooth)
}
\references{
\cite{Rou Zhong, Shishi Liu, Haocheng Li, Jingxiao Zhang (2021). "Robust Functional Principal Component Analysis for Non-Gaussian Longitudinal Data." Journal of Multivariate Analysis, https://doi.org/10.1016/j.jmva.2021.104864.}
}
