# GompertzExt.R
#' Gompertz Extension Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Gompertz Extension distribution.
#'
#' The Gompertz Extension distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Gompertz Extension distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda, \theta) =
#' \quad 1-\exp \left\{-\lambda\left(e^{\alpha x}-1\right)^\theta\right\}
#'  \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The functions available are listed below:
#' \itemize{
#'   \item \code{dgompertz.ext()} — Density function
#'   \item \code{pgompertz.ext()} — Distribution function
#'   \item \code{qgompertz.ext()} — Quantile function
#'   \item \code{rgompertz.ext()} — Random generation
#'   \item \code{hgompertz.ext()} — Hazard function
#' }
#'
#' @usage
#' dgompertz.ext(x, alpha, lambda, theta, log = FALSE)
#' pgompertz.ext(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qgompertz.ext(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rgompertz.ext(n, alpha, lambda, theta)
#' hgompertz.ext(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dgompertz.ext}: numeric vector of (log-)densities
#'   \item \code{pgompertz.ext}: numeric vector of probabilities
#'   \item \code{qgompertz.ext}: numeric vector of quantiles
#'   \item \code{rgompertz.ext}: numeric vector of random variates
#'   \item \code{hgompertz.ext}: numeric vector of hazard values
#' }
#'
#' @references 
#' Chaudhary, A.K., & Kumar, V. (2020). 
#' A Bayesian Estimation and Prediction of Gompertz Extension Distribution 
#' Using the MCMC Method. \emph{Nepal Journal of Science and Technology(NJST)},
#' \bold{19(1)}, 142--160. \doi{10.3126/njst.v19i1.29795}

#'
#' @examples
#' x <- seq(1.0, 10, 0.25)
#' dgompertz.ext(x, 0.1, 5.0, 2.5)
#' pgompertz.ext(x, 0.1, 5.0, 2.5)
#' qgompertz.ext(0.5, 0.1, 5.0, 2.5)
#' rgompertz.ext(10, 0.1, 5.0, 2.5)
#' hgompertz.ext(x, 0.1, 5.0, 2.5)
#' 
#' # Data
#' x <- stress
#' # ML estimates
#' params = list(alpha=0.0678, lambda=44.34760, theta=2.5225)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pgompertz.ext, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qgompertz.ext, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dgompertz.ext, pfun=pgompertz.ext, plot=TRUE)
#' print.gofic(out)
#' 
#' @name GompertzExt
#' @aliases dgompertz.ext pgompertz.ext qgompertz.ext rgompertz.ext hgompertz.ext
#' @export
NULL

#' @rdname GompertzExt
#' @usage NULL
#' @export   
dgompertz.ext <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
    u <- exp(alpha * x[valid])
	tmp <- exp(-lambda * ((u -1) ^ theta)) 
    pdf[valid] <- alpha * theta * lambda * u * ((u -1) ^ (theta-1)) * tmp
    } 
	if (log) 
        pdf <- log(pdf)
    return(pdf)   
}

#' @rdname GompertzExt
#' @usage NULL
#' @export  
pgompertz.ext <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- exp(q[valid] * alpha)
	tmp <- exp(-lambda * ((u -1) ^ theta))
    cdf[valid] <- 1.0 - tmp                   
    }
	if (!lower.tail) 
        cdf <- 1.0 - cdf 
    if (log.p) 
        cdf <- log(cdf)    
    return(cdf)   
}

#' @rdname GompertzExt
#' @usage NULL
#' @export  
qgompertz.ext <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda' and 'theta' must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
    qtl[valid] <- (1.0/alpha) * log(1.0 + ((-1/lambda) * log(1.0 - p[valid])) ^(1/theta))      
    }     
    return(qtl)   
}

#' @rdname GompertzExt
#' @usage NULL
#' @export 
rgompertz.ext <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qgompertz.ext(u, alpha, lambda, theta)
    return(q)
}

#' @rdname GompertzExt
#' @usage NULL
#' @export 
hgompertz.ext <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
	nume <- dgompertz.ext(x, alpha, lambda, theta) 
	surv <- 1.0 - pgompertz.ext(x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)   
}


