#' @title OVL.GPQ
#' @description Parametric approach based on generalized inference.
#' @param x Numeric vector. Data from the first group.
#' @param y Numeric vector. Data from the second group.
#' @param alpha confidence level.
#' @param K Number of simulated generalized pivotal quantities.
#' @param h_ini initial value in the optimization problem.
#' @param BC Logical. Indicates whether a Box–Cox transformation is applied to the data.
#' @return confidence interval.
#' @export OVL.GPQ
#' @importFrom stats optim pnorm quantile rt rchisq sd
#' @examples
#' controls = rnorm(50,6,1)
#' cases = rnorm(100,6.5,0.5)
#' OVL.GPQ (controls,cases)


OVL.GPQ<-function(x,y,alpha=0.05,K=2500,h_ini=-1.6,BC=FALSE){
  xo<-x
  yo<-y
  if(BC==TRUE){
    x_aux<-x
    y_aux<-y
    all_values<-c(x_aux,y_aux)
    if (any(all_values<=0)){
      x<-x_aux+abs(min(all_values))+(max(all_values)-min(all_values))/2
      y<-y_aux+abs(min(all_values))+(max(all_values)-min(all_values))/2
    } else {
      x<-x_aux
      y<-y_aux
    }
    xo<-x
    yo<-y

    hhat_BFGS <- try(
      { optim(h_ini, likbox, data = c(xo, yo), n = length(xo), method = "BFGS")$par },
      silent = TRUE
    )

    if (inherits(hhat_BFGS, "try-error")) {

      hhat_LBFGSB <- try(
        { optim(h_ini, likbox, data = c(xo, yo), n = length(xo),
                method = "L-BFGS-B", lower = -2, upper = 2)$par },
        silent = TRUE
      )

      if (inherits(hhat_LBFGSB, "try-error")) {

        hhat_NM <- try(
          { optim(h_ini, likbox, data = c(xo, yo), n = length(xo),
                  method = "Nelder-Mead")$par },
          silent = TRUE
        )

        if (inherits(hhat_NM, "try-error")) {
          hhat <- h_ini
        } else {
          hhat <- hhat_NM
        }

      } else {
        hhat <- hhat_LBFGSB
      }

    } else {
      hhat <- hhat_BFGS
    }


    if (abs(hhat)<1e-5){
      x<-log(xo)
      y=log(yo)
    }else{
      x<-((xo^hhat)-1)/hhat
      y<-((yo^hhat)-1)/hhat
    }
  }else{
    x<-xo
    y<-yo
  }

  mu1_hat<-mean(x)
  mu2_hat<-mean(y)
  e_x<-sd(x)
  e_y<-sd(y)
  R_OVL<-numeric(K)

  for(i in 1:K){
    t_x<-rt(1,df=length(x)-1)
    t_y<-rt(1,df=length(y)-1)
    ji_x<-rchisq(1,df=length(x)-1)
    ji_y<-rchisq(1,df=length(y)-1)
    R_mu_x<-mu1_hat-t_x*e_x/sqrt(length(x))
    R_mu_y<-mu2_hat-t_y*e_y/sqrt(length(y))
    R_var_x<-(1/ji_x)*(length(x)-1)*(e_x)^2
    R_var_y<-(1/ji_y)*(length(y)-1)*(e_y)^2
    R_sigma_x<-sqrt(R_var_x)
    R_sigma_y<-sqrt(R_var_y)
    if(R_sigma_y<R_sigma_x){
      auxmu<-R_mu_x
      auxsigma<-R_sigma_x
      R_mu_x<-R_mu_y
      R_mu_y<-auxmu
      R_sigma_x<-R_sigma_y
      R_sigma_y<-auxsigma
    }
    R_x1<-((R_mu_x*R_sigma_y^2-R_mu_y*R_sigma_x^2)-R_sigma_x*R_sigma_y*sqrt((R_mu_x-R_mu_y)^2+(R_sigma_x^2-R_sigma_y^2)*log(R_sigma_x^2/R_sigma_y^2)))/(R_sigma_y^2-R_sigma_x^2)
    R_x2<-((R_mu_x*R_sigma_y^2-R_mu_y*R_sigma_x^2)+R_sigma_x*R_sigma_y*sqrt((R_mu_x-R_mu_y)^2+(R_sigma_x^2-R_sigma_y^2)*log(R_sigma_x^2/R_sigma_y^2)))/(R_sigma_y^2-R_sigma_x^2)
    R_OVL[i]<-1+pnorm((R_x1-R_mu_x)/R_sigma_x)-pnorm((R_x1-R_mu_y)/R_sigma_y)-pnorm((R_x2-R_mu_x)/R_sigma_x)+pnorm((R_x2-R_mu_y)/R_sigma_y)
  }
  IC1<-quantile(R_OVL,alpha/2,na.rm=TRUE)
  IC2<-quantile(R_OVL,1-alpha/2,na.rm=TRUE)
  return(list(IC1=IC1,IC2=IC2))
}
