#' Calculate Log-Likelihood for DGLM
#'
#' Element-wise log-likelihood for Dynamic Generalized Linear Models
#' with linear predictor eta = X * beta.
#'
#' @param y Numeric response vector (length n).
#' @param X Numeric design matrix (n x p).
#' @param beta Numeric coefficient vector (length p).
#' @param family Character distribution choice: "poisson", "pareto", or "gamma".
#' @param alpha_gamma Shape parameter for Gamma (default 2).
#' @param lambda_min Lower bound for Pareto shape lambda (default 1e-6).
#'
#' @return Numeric vector of length n; -Inf for illegal observations.
#'
#' @examples
#' set.seed(123)
#' X <- matrix(rnorm(100 * 3), 100, 3)
#' beta <- c(0.5, -0.2, 0.1)
#'
#' # Poisson
#' y_pois <- rpois(100, lambda = exp(X %*% beta))
#' ll_pois <- dglm_likelihood(y_pois, X, beta, family = "poisson")
#'
#' # Gamma
#' y_gamma <- rgamma(100, shape = 3, rate = exp(X %*% beta))
#' ll_gamma <- dglm_likelihood(y_gamma, X, beta, family = "gamma", alpha_gamma = 3)
#'
#' @export
dglm_likelihood <- function(y, X, beta,
                            family = c("poisson", "pareto", "gamma"),
                            alpha_gamma = 2,
                            lambda_min = 1e-6) {
  family <- match.arg(family)
  eta <- as.numeric(X %*% beta)  # linear predictor
  
  switch(family,
         poisson = {
           lambda <- pmax(exp(eta), .Machine$double.eps)
           stats::dpois(y, lambda = lambda, log = TRUE)
         },
         pareto = {
           # inverse link: eta = log(lambda/(lambda-1)) => lambda = 1/(1 - exp(-eta))
           lambda <- 1 / pmax(1 - exp(-eta), lambda_min)
           ifelse(y >= 1,
                  log(lambda) - (lambda + 1) * log(y),
                  -Inf)
         },
         gamma = {
           rate <- pmax(exp(eta), .Machine$double.eps)
           stats::dgamma(y, shape = alpha_gamma, rate = rate, log = TRUE)
         })
}

