% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hcbeta.R
\name{hc.beta}
\alias{hc.beta}
\title{Compute Revelle's Beta for all worst split-halves using HC-SVD.}
\usage{
hc.beta(R, splits = NULL, n.splits = NULL, is.corr = TRUE, verbose = TRUE)
}
\arguments{
\item{R}{A correlation matrix of dimension \eqn{p}x\eqn{p} or a data matrix of dimension \eqn{n}x\eqn{p} can be provided. If a data matrix is supplied, it must be indicated by setting
\code{is.corr = FALSE}, and the correlation matrix will then be calculated as \code{cor(X)}.}

\item{splits}{An object containing the splits identified by HC-SVD. This can either be the result of \code{\link{hcsvd}} (for all splits)
or \code{\link{bd.approx}} (for a single split). If omitted, \code{hc.beta} will internally call \code{hcsvd(R)} and
compute Revelle's beta for all \eqn{p - 1} splits.}

\item{n.splits}{Number of splits for which Revelle's beta is computed. If \code{splits} is from \code{\link{hcsvd}}, the default is all \eqn{p-1}
splits. If \code{splits} is from \code{\link{bd.approx}}, only a single split is available and \code{n.splits} is set to 1.}

\item{is.corr}{Is the supplied object a correlation matrix. Default is \code{TRUE} and this parameter must be set to \code{FALSE} if
a data matrix instead of a correlation matrix is supplied.}

\item{verbose}{Print out progress as \eqn{p-1} iterations for divisive hierarchical clustering are performed.
Default is \code{TRUE}.}
}
\value{
A list with \code{n.splits} components. Each split is a list of four components:
\item{split}{
The split number.
}
\item{beta}{
Revelle's beta for this split.
}
\item{A}{
One of the two sub-scales that has been split.
}
\item{B}{
One of the two sub-scales that has been split.
}
\item{beta.alpha}{
Computes the ratio of Revelle's beta and Cronbach's alpha.
}
}
\description{
Performs HC-SVD to reveal the hierarchical variable structure using average linkage as described in Bauer (202Xa). For a data matrix comprising \eqn{p} items,
this means that \eqn{p-1} splits are identified. The obtained structure aligns with the structure according to the worst split-half reliability and is thus used to compute
a hierarchy of all Revelle's beta as described in Bauer (202Xb).
}
\details{
Supplementary details are in Bauer (202Xb).
}
\examples{
#We compute the worst split-half reliabilities on a correlation matrix.

\donttest{
#Load the correlation matrix Bechtoldt from the psych
#package (see ?Bechtoldt for more information).
if (requireNamespace("psych", quietly = TRUE)) {
  data("Bechtoldt", package = "psych")
}
R <- Bechtoldt


### RUN HC-SVD FOR HIERARCHICAL VARIABLE CLUSTERING

#Compute HC-SVD (with average linkage).
hcsvd.obj <- hcsvd(R)

#The object of type hclust with corresponding dendrogram can be obtained
#directly from hcsvd(...):
hc.div <- hcsvd.obj$hclust
plot(hc.div, axes = FALSE, ylab = "", main = "Revelle's Beta Splits")


### COMPUTE REVELLE'S BETA FOR ALL IDENTIFIED SPLITS

#Compute Revelle's beta
betas <- hc.beta(R = R)

#Alternatively, you can submit the object obtained from hcsvd(). Thus,
#the hiearchy needs not to be computed again using hcsvd().
betas <- hc.beta(R = R, splits = hcsvd.obj)

#Visualize the splits, e.g., as
splits <- sapply(betas, `[[`, "split")
beta.values <- sapply(betas, `[[`, "beta")

plot(splits, beta.values,
  type = "b",
  xlab = "Split",
  ylab = "Revelle's Beta",
  main = "Revelle's Beta Across Splits",
  pch = 19)

#Visualize the ratio of Revelle's beta and Cronbach's alpha
beta.alpha <- sapply(betas, `[[`, "beta.alpha")
plot(splits, beta.values,
  type = "b",
  xlab = "Split",
  ylab = "Beta/Alpha",
  main = "Ratio of Beta and Alpha Across Splits",
  pch = 19)


### COMPUTE REVELLE'S BETA FOR THE FIRST IDENTIFIED SPLIT

#The first split can be identified using bd.approx()
#This is computationally faster, as only the first split
#is identified
hc.beta(R = R, splits = bd.approx(R))

}


}
\references{
\cite{Bauer, J.O. (202Xa). Divisive hierarchical clustering using block diagonal matrix approximations. Working paper.}

\cite{Bauer, J.O. (202Xb). Revelle's beta: The wait is over - we can compute it!. Working paper.}
}
\seealso{
\code{\link{bd.approx}} \code{\link{hcsvd}}
}
