\name{catch}
\alias{catch}
\title{
Fit a CATCH model and predict categorical response.
}
\description{
The \code{catch} function solves classification problems and selects variables by fitting a covariate-adjusted tensor classification in high-dimensions (CATCH) model. The input training predictors include two parts: tensor data and low dimensional covariates. The tensor data could be matrix as a special case of tensor. In \code{catch}, tensor data should be stored in a list form. If the dataset contains no covariate, \code{catch} can also fit a classifier only based on the tensor predictors. If covariates are provided, the method will adjust tensor for covariates and then fit a classifier based on the adjusted tensor along with the covariates. If users specify testing data at the same time, predicted response will be obtained as well. 
}
\usage{
catch(x, z = NULL, y, testx = NULL, testz = NULL, nlambda = 100, 
lambda.factor = ifelse((nobs - nclass) <= nvars, 0.2, 1E-03), 
lambda = NULL,dfmax = nobs, pmax = min(dfmax * 2 + 20, nvars), 
pf = rep(1, nvars), eps = 1e-04, maxit = 1e+05, sml = 1e-06, 
verbose = FALSE, perturb = NULL)
}
\arguments{
  \item{x}{Input tensor (or matrix) list of length \eqn{N}{N}, where \eqn{N}{N} is the number of observations. Each element of the list is a tensor or matrix. The order of tensor can be any positive integer not less than 2. }
  \item{z}{Input covariate matrix of dimension \eqn{N \times q}{N*q}, where \eqn{q<N}{q<N}. \code{z} can be omitted if covariate is absent. }
  \item{y}{Class label. For \code{K} class problems, \code{y} takes values in \eqn{\{1,\cdots,\code{K}\}}.}
  \item{testx}{Input testing tensor or matrix list. Each element of the list is a test case. When \code{testx} is not provided, the function will only fit the model and return the classifier. When \code{testx} is provided, the function will predict response on \code{testx} as well.}
  \item{testz}{Input testing covariate matrix. Can be omitted if covariate is absent. However, training covariates \code{z} and testing covariates \code{testz} must be provided or not at the same time.}
  \item{nlambda}{The number of tuning values in sequence \code{lambda}. If users do not specify \code{lambda} values, the package will generate a solution path containing \code{nlambda} many tuning values of \code{lambda}. Default is \code{100}. }
  \item{lambda.factor}{When \code{lambda} is not supplied, \code{catch} first finds the largest value in \code{lambda} which yields \eqn{\boldsymbol{\beta}=0}. Then the minimum value in \code{lambda} is obtained by \code{(largest value*lambda.factor)}. The sequence of \code{lambda} is generated by evenly sampling \code{nlambda} numbers within the range. Default value of \code{lambda.factor} is 0.2 if \eqn{N<p}{N<p} and 0.0001 if \eqn{N>p}{N>p}.}
  \item{lambda}{A sequence of user-specified \code{lambda} values. \code{lambda} is the weight of L1 penalty and a smaller \code{lambda} allows more variables to be nonzero. If \code{NULL}, then the algorithm will generate a sequence of \code{nlambda} many potential 
\code{lambda}s according to \code{lambda.factor}.}
  \item{dfmax}{The maximum number of selected variables in the model. Default is the number of observations \code{N}.}
  \item{pmax}{The maximum number of potential selected variables during iteration. In middle step, the algorithm can select at most \code{pmax} variables and then shrink part of them such that the nubmer of final selected variables is less than \code{dfmax}. Default is \eqn{\min(dfmax\times 2+20, N)}{min (dfmax*2+20, p)}.
  }
  \item{pf}{Weight of lasso penalty. Default is a vector of value \code{1} and length \code{p}, representing L1 penalty of length \eqn{p}{p}. Can be mofidied to use adaptive lasso penalty.}
  \item{eps}{Convergence threshold for coordinate descent difference between iterations. Default value is \code{1e-04}.}
  \item{maxit}{Maximum iteration times for all lambda. Default value is \code{1e+05}.}
  \item{sml}{Threshold for ratio of loss function change after each iteration to old loss function value. Default value is \code{1e-06}.}
  \item{verbose}{Indicates whether print out lambda during iteration or not. Default value is \code{FALSE}.}
  \item{perturb}{Perturbation scaler. If it is specified, the value will be added to diagonal of estimated covariance matrix. A small value can be used to accelerate iteration. Default value is \code{NULL}.
  }
}
\details{The \code{\link{catch}} function fits a linear discriminant analysis model as follows:
\deqn{\mathbf{Z}|(Y=k)\sim N(\boldsymbol{\phi_k},\boldsymbol{\psi}),}
\deqn{\mathbf{X}|(\mathbf{Z}=\mathbf{z}, Y=k)\sim TN(\boldsymbol{\mu}_k+\boldsymbol{\alpha}\bar{\times}_{M+1}\mathbf{z},\boldsymbol{\Sigma}_1,\cdots,\boldsymbol{\Sigma}_M).}
The categorical response is predicted from the estimated Bayes rule:
\deqn{\widehat{Y}=\arg\max_{k=1,\cdots,K}{a_k+\boldsymbol{\gamma}_k^T\mathbf{Z}+<\boldsymbol{\beta}_k,\mathbf{X}-\boldsymbol{\alpha}\overline{\times}_{M+1}\mathbf{Z}>},} 
where \eqn{\mathbf{X}} is the tensor, \eqn{\mathbf{Z}} is the covariates, \eqn{a_k}, \eqn{\boldsymbol{\gamma}_k} and \eqn{\boldsymbol{\alpha}} are parameters estimated by CATCH. A detailed explanation can be found in reference. When \code{Z} is not \code{NULL}, the function will first adjust tensor on covariates by modeling
\deqn{\mathbf{X}=\boldsymbol{\mu}_k+\boldsymbol{\alpha}\overline{\times}_{M+1}\mathbf{Z}+\mathbf{E},} where \eqn{\mathbf{E}} is an unobservable tensor normal error independent of \eqn{\mathbf{Z}}. 
Then \code{\link{catch}} fits model on the adjusted training tensor \eqn{\mathbf{X}-\boldsymbol{\alpha}\overline{\times}_{M+1}\mathbf{Z}} and makes predictions on testing data by using the adjusted tensor list. If \code{Z} is \code{NULL}, it reduces to a simple tensor discriminant analysis model. 

The coefficient of tensor \eqn{\boldsymbol{\beta}}, represented by \code{beta} in package, is estimated by 
\deqn{\min_{\boldsymbol{\beta}_2,\ldots,\boldsymbol{\beta}_K}\left[\sum_{k=2}^K\left(\langle\boldsymbol{\beta}_k,[\![\boldsymbol{\beta}_k;\widehat{\boldsymbol{\Sigma}}_{1},\dots,\widehat{\boldsymbol{\Sigma}}_{M}]\!]\rangle-2\langle\boldsymbol{\beta}_k,\widehat{\boldsymbol{\mu}}_{k}-\widehat{\boldsymbol{\mu}}_{1}\rangle\right)+\lambda\sum_{j_{1}\dots j_{M}}\sqrt{\sum_{k=2}^{K}\beta_{k,j_{1}\cdots j_{M}}^2}\right].}
When response is multi-class, the group lasso penalty over categories is added to objective function through parameter \code{lambda}, and it reduces to a lasso penalty in binary problems.

The function \code{\link{catch}} will predict categorical response when testing data is provided. 
If testing data is not provided or if one wishes to perform prediction separately, \code{\link{catch}} can be used to only fit model with a catch object outcome. The object outcome can be combined with the adjusted tensor list from \code{\link{adjten}} to perform prediction by \code{\link{predict.catch}}.
}
\value{
\item{beta}{Output variable coefficients for each \code{lambda}, which is the estimation of \eqn{\boldsymbol{\beta}} in the Bayes rule. \code{beta} is a list of length being the number of \code{lambda}s. Each element of \code{beta} is a matrix of dimension \eqn{nvars\times (nclass-1)}{nvars*(nclass-1)}.}
\item{df}{The number of nonzero variables for each value in sequence \code{lambda}.}
\item{dim}{Dimension of coefficient array.}
\item{lambda}{The actual \code{lambda} sequence used. The user specified sequence or automatically generated sequence could be truncated by constraints on \code{dfmax} and \code{pmax}.}
\item{obj}{Objective function value for each value in sequence \code{lambda}.}
\item{x}{The tensor list after adjustment in training data. If covariate is absent, this is the original input tensor list.}
\item{y}{Class label in training data.}
\item{npasses}{Total number of iterations.}
\item{jerr}{Error flag.}
\item{sigma}{Estimated covariance matrix on each mode. \code{sigma} is a list with the \code{i}th element being covariance matrix on \code{i}th mode.}
\item{delta}{Estimated delta matrix \eqn{(vec(\widehat{\boldsymbol{\mu}}_2-\widehat{\boldsymbol{\mu}}_1),\cdots,vec(\widehat{\boldsymbol{\mu}}_K-\widehat{\boldsymbol{\mu}}_1))}.}
\item{mu}{Estimated mean array of \eqn{\mathbf{X}}.}
\item{prior}{Proportion of samples in each class.}
\item{call}{The call that produces this object.}
\item{pred}{Predicted categorical response for each value in sequence \code{lambda} when \code{testx} is provided.}
}
\references{
Pan, Y., Mai, Q., and Zhang, X. (2018) \emph{Covariate-Adjusted Tensor Classification in High-Dimensions}, arXiv:1805.04421.
}
\author{
Yuqing Pan, Qing Mai, Xin Zhang
}
\seealso{
\code{\link{cv.catch}}, \code{\link{predict.catch}}, \code{\link{adjten}}
}
\examples{
#without prediction
n <- 20
p <- 4
k <- 2
nvars <- p*p*p
x <- array(list(),n)
vec_x <- matrix(rnorm(n*nvars), nrow=n, ncol=nvars)
vec_x[1:10,] <- vec_x[1:10,]+2
z <- matrix(rnorm(n*2), nrow=n, ncol=2)
z[1:10,] <- z[1:10,]+0.5
y <- c(rep(1,10),rep(2,10))
for (i in 1:n){
  x[[i]] <- array(vec_x[i,],dim=c(p,p,p))
}
obj <- catch(x,z,y=y)
}