#'
#' @title Checks if an object is defined on the server-side
#' @description Looks if an R object of the given name is defined on the server-side. 
#' This function is similar to the R function \code{exists}. 
#' @details In DataSHIELD it is not possible to see the data on the servers
#' of the collaborating studies. It is only possible to get summaries of objects stored on the
#' server-side. 
#' It is however important to know if an object is defined (i.e. exists) on the server-side.
#' This function checks if an object does exist on the server-side.
#' 
#' Server function called: \code{exists}
#' @param x a character string providing the name of the object to look for.
#' @param datasources a list of \code{\link[DSI]{DSConnection-class}} 
#' objects obtained after login. If the \code{datasources} argument is not specified
#' the default set of connections will be used: see \code{\link[DSI]{datashield.connections_default}}.
#' @return \code{ds.exists} returns a logical object. 
#' TRUE if the object is on the server-side and FALSE otherwise.
#' @author DataSHIELD Development Team
#' @seealso \code{\link{ds.class}} to check the type of an object.
#' @seealso \code{\link{ds.length}} to check the length of an object.
#' @seealso \code{\link{ds.dim}} to check the dimension of an object.
#' @export
#' @examples
#' \dontrun{
#'
#'   ## Version 6, for version 5 see the Wiki
#'   
#'   # connecting to the Opal servers
#' 
#'   require('DSI')
#'   require('DSOpal')
#'   require('dsBaseClient')
#'
#'   builder <- DSI::newDSLoginBuilder()
#'   builder$append(server = "study1", 
#'                  url = "http://192.168.56.100:8080/", 
#'                  user = "administrator", password = "datashield_test&", 
#'                  table = "CNSIM.CNSIM1", driver = "OpalDriver")
#'   builder$append(server = "study2", 
#'                  url = "http://192.168.56.100:8080/", 
#'                  user = "administrator", password = "datashield_test&", 
#'                  table = "CNSIM.CNSIM2", driver = "OpalDriver")
#'   builder$append(server = "study3",
#'                  url = "http://192.168.56.100:8080/", 
#'                  user = "administrator", password = "datashield_test&", 
#'                  table = "CNSIM.CNSIM3", driver = "OpalDriver")
#'   logindata <- builder$build()
#'   
#'   connections <- DSI::datashield.login(logins = logindata, assign = TRUE, symbol = "D") 
#'   
#'   # Check if the object exist in the server-side
#'   ds.exists(x = "D", 
#'             datasources = connections) #All opal servers are used
#'   ds.exists(x = "D", 
#'             datasources = connections[1]) #Only the first Opal server is used (study1)
#'             
#'   # clear the Datashield R sessions and logout
#'   datashield.logout(connections)
#'
#' }
#'
ds.exists <- function(x=NULL, datasources=NULL){

  # look for DS connections
  if(is.null(datasources)){
    datasources <- datashield.connections_find()
  }

  # ensure datasources is a list of DSConnection-class
  if(!(is.list(datasources) && all(unlist(lapply(datasources, function(d) {methods::is(d,"DSConnection")}))))){
    stop("The 'datasources' were expected to be a list of DSConnection-class objects", call.=FALSE)
  }

  if(is.null(x)){
    stop("Please provide the name of the input object!", call.=FALSE)
  }

  # call the server side function that does the job
  cally <- call("exists", x)
  output <- DSI::datashield.aggregate(datasources, cally)

  return(output)
}
