% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ufRisk-functions.R, R/distr_est.R
\name{measure_risk}
\alias{measure_risk}
\alias{measure_risk,fEGarch_fit-method}
\alias{measure_risk,fEGarch_forecast-method}
\alias{measure_risk,fEGarch_distr_est-method}
\title{VaR and ES Computation Following Fitted Models or Forecasts}
\usage{
measure_risk(object, measure = c("VaR", "ES"), level = c(0.975, 0.99), ...)

\S4method{measure_risk}{fEGarch_fit}(object, measure = c("VaR", "ES"), level = c(0.975, 0.99), ...)

\S4method{measure_risk}{fEGarch_forecast}(object, measure = c("VaR", "ES"), level = c(0.975, 0.99), ...)

\S4method{measure_risk}{fEGarch_distr_est}(
  object,
  measure = c("VaR", "ES"),
  level = c(0.975, 0.99),
  test_obs,
  sigt,
  cmeans,
  ...
)
}
\arguments{
\item{object}{either an object of class \code{"fEGarch_fit"} returned by the
fitting / estimation functions of this package like returned by for example
\code{\link{fEGarch}} among others, an object of class \code{"fEGarch_forecast"}
as returned by \code{\link{predict,fEGarch_fit-method}} or
\code{\link{predict_roll,fEGarch_fit-method}}, or an object of class \code{"fEGarch_distr_est"} returned by the
distribution fitting functions of this package like returned by for example
\code{\link{find_dist}} among others.}

\item{measure}{a character vector with element \code{"VaR"}, \code{"ES"} or both;
indicates, what risk measures should be computed; by default, both VaR and ES
are computed.}

\item{level}{a numeric vector of arbitrary length indicating the confidence
levels to compute the VaR and the ES at; by default, the levels
\code{0.975} and \code{0.99}, i.e. 97.5 percent and 99 percent, are considered.}

\item{...}{currently without use.}

\item{test_obs}{a series of test observations (only required when \code{object} is of class \code{"fEGarch_distr_est"}).}

\item{sigt}{a series of forecasted conditional standard deviations for the
same time points as \code{test_obs} (only required when \code{object} is of class \code{"fEGarch_distr_est"}).}

\item{cmeans}{a series of forecasted conditional means for the
same time points as \code{test_obs} (only required when \code{object} is of class \code{"fEGarch_distr_est"}).}
}
\value{
The S4 methods all return an object of class \code{"fEGarch_risk"} with elements \code{measures},
\code{observations} and \code{model}.
\code{observations} is the observed series at the time points, for which the
risk measures are calculated. \code{measures} is a list with elements
\code{VaR} and \code{ES}, distinguishing between computed VaR and ES values.
These elements again are list with named elements representing the various
computed series. \code{model} is the fitted model or a list of multiple fitted model objects.
}
\description{
Provides easy access to value-at-risk (VaR) and expected shortfall (ES)
computation for available models in this package. VaR and ES can either
be computed based on (a) fitted conditional means and conditional
standard deviations for a training period, or following (b) point forecasts
(one-step rolling) of the conditional means and conditional
standard deviations.
}
\details{
Given a fitted model with fitted conditional means and conditional standard deviations
or given point forecasts of such series based on a fitted model, the risk measures
VaR and ES can be computed (at arbitrary confidence levels) following the conditional
loss distribution defined through the estimated / forecasted conditional mean value,
the estimated / forecasted conditional standard deviation value, and the assumed
conditional distribution (including potential estimates of distribution parameters).

Let \eqn{\hat{\mu}_t} be the estimated / forecasted conditional mean and \eqn{\hat \sigma_t} be the
estimated / forecasted conditional standard deviation at some time point \eqn{t}. Furthermore,
define \eqn{\text{VaR}_{\eta,\alpha}} and \eqn{\text{ES}_{\eta,\alpha}} be
the time-invariant VaR and ES, respectively, of some identically but independently
distributed random variables \eqn{\eta_t} with mean zero and variance one. Given that
the relationship \eqn{r_t = \mu_t + \sigma_t\eta_t}, where \eqn{\mu_t} and \eqn{\sigma_t}
are the true conditional mean and conditional standard deviation at time \eqn{t}, is assumed
for some return series \eqn{\{r_t\}}, the estimated / forecasted conditional VaR and ES of \eqn{r_t} are simply
\deqn{\widehat{\text{VaR}}_{r,\alpha}(t) = \hat{\mu}_t + \hat{\sigma}_t \text{VaR}_{\eta,\alpha}  \hspace{3mm} \text{ and } \hspace{3mm} \widehat{\text{ES}}_{r,\alpha}(t) = \hat{\mu}_t + \hat{\sigma}_t \text{ES}_{\eta,\alpha}.}
This definition holds, when losses and therefore also
\eqn{\text{VaR}_{\eta,\alpha}(t)} and \eqn{\text{ES}_{\eta,\alpha}(t)} (for common \eqn{\alpha} such as
\eqn{\alpha = 0.975} or \eqn{\alpha = 0.99}) are considered
to be negative in sign.

Define
\deqn{\text{VaR}_{\eta,\alpha} = f_{\eta}^{-1}(1-\alpha) \hspace{3mm} \text{ and } \hspace{3mm} \text{ES}_{\eta,\alpha} = (1-\alpha)^{-1}\int_{\alpha}^{1} \text{VaR}_{\eta, x} dx,}
which also need to be estimated for some distributions, if a distribution parameter needed to be estimated. \eqn{f} in the previous formula
is the cumulative distribution function of the random variables \eqn{\eta_t}. Therefore,
\eqn{f^{-1}_{\eta}(1-\alpha)} returns the quantile of the innovation distribution at level
\eqn{1-\alpha}.

In some cases, when rolling
one-step forecasts of the conditional standard deviations and the conditional means
were obtained following a nonparametric approach, for example through
neural networks or similar approaches, VaR and ES are not directly to be calculated
because distribution assumptions have not been made. If an \code{object} that
is a fitted distribution to the model's standardized in-sample residuals is provided,
and if also test observations as well as forecasted conditional standard deviations
and conditional means for the test time points are passed to the method, VaR
and ES will be computed using the fitted distribution in \code{object}. Note
that \code{object} must be of class \code{"fEGarch_distr_est"}. A natural
selection of \code{object} is the output of \code{\link{find_dist}}, which returns
the best fitted model among a normal distribution, a \eqn{t}-distribution, a
generalized error distribution, an average Laplace distribution, and their skewed
variants, following either BIC (the default) or AIC. It is recommended to then
set \code{fix_mean = 0} and \code{fix_sdev = 1} in the call to
\code{\link{find_dist}} to reflect the known property that the residuals are assumed
to be estimated from innovations with mean zero and variance one.
}
\examples{

# In-sample
window.zoo <- get("window.zoo", envir = asNamespace("zoo"))
rt <- window.zoo(SP500, end = "2002-12-31")
model <- fEGarch(egarch_spec(), rt)
risk <- measure_risk(model, measure = c("VaR", "ES"), level = c(0.95, 0.975, 0.99))
risk

# Out-of-sample rolling point forecasts
window.zoo <- get("window.zoo", envir = asNamespace("zoo"))
rt <- window.zoo(SP500, end = "2002-12-31")
model2 <- fEGarch(egarch_spec(), rt, n_test = 250)
# parallel = FALSE only for better compatibility with CRAN checks;
# should be set to TRUE for speed boost
fcast <- predict_roll(model2, parallel = FALSE)
risk2 <- measure_risk(fcast, measure = c("VaR", "ES"), level = c(0.95, 0.975, 0.99))
risk2

# Use some model to obtain rolling point forecasts of
# the conditional mean and the conditional standard deviation for
# some test period; in practice, this will not be from a GARCH-type
# model, because it is parametric and includes a distribution assumption,
# but instead from some nonparametric model
window.zoo <- get("window.zoo", envir = asNamespace("zoo"))
rt <- window.zoo(SP500, end = "2005-12-31")
model <- fEGarch(egarch_spec(), rt, n_test = 250)
# parallel = FALSE only for better compatibility with CRAN checks;
# should be set to TRUE for speed boost
fc <- model \%>\% predict_roll(refit_after = 50, parallel = FALSE)

test_obs <- model@test_obs   # Test observations
sigt <- fc@sigt              # Conditional volatility forecasts
cmeans <- fc@cmeans          # Conditional mean forecasts

resids <- model@etat         # In-sample standardized residuals

# Given 'test_obs', 'sigt', 'cmeans' and 'resids', we can now
# compute the VaR and ES forecasts for the test period

dist <- find_dist(resids, fix_mean = 0, fix_sdev = 1)
dist

risk <- dist \%>\%
 measure_risk(test_obs = test_obs, sigt = sigt, cmeans = cmeans)

plot(risk, which = 0.975)



}
