% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glmmFEL.R
\name{glmmFEL}
\alias{glmmFEL}
\title{Fit GLMMs via Laplace and fully exponential Laplace (matrix interface)}
\usage{
glmmFEL(
  y,
  X,
  Z,
  family = stats::binomial(link = "probit"),
  approx = c("FE", "Laplace", "FE_mean", "FE_full", "RSPL", "MSPL"),
  max_iter = 200,
  tol = 1e-06,
  control = list()
)
}
\arguments{
\item{y}{Numeric response vector of length \eqn{n}. For
\code{family = "binomial_probit"} / \code{binomial(link = "probit")} or
\code{family = "binomial_logit"} / \code{binomial(link = "logit")},
values must be 0 or 1.}

\item{X}{Fixed-effects design matrix of dimension \eqn{n \times p}. May be a
base R matrix or a matrix-like object; it is internally coerced to a
base numeric matrix. Must have full column rank.}

\item{Z}{Random-effects design matrix of dimension \eqn{n \times q}. May be a
base R matrix or a \pkg{Matrix} object. Internally it is coerced to a
sparse \code{"dgCMatrix"} where possible (to preserve sparsity).
Must have at least one column (no purely fixed-effects models).}

\item{family}{Either a character string or a \link[stats:family]{stats::family} object indicating the
model family. The argument is resolved via \code{\link[=glmmfe_resolve_family]{glmmfe_resolve_family()}}.}

\item{approx}{Approximation type, resolved via \code{\link[=glmmfe_resolve_approx]{glmmfe_resolve_approx()}}. Accepted
values (case-insensitive) include:
\itemize{
\item \code{"Laplace"} – first-order Laplace approximation,
\item \code{"FE_mean"} – staged algorithm: Laplace phase then FE mean corrections,
\item \code{"FE"} / \code{"FE_full"} – staged algorithm: Laplace phase, then FE mean,
then FE covariance corrections (default),
\item \code{"RSPL"} – restricted pseudo-likelihood (REML-style) linearization,
\item \code{"MSPL"} – marginal pseudo-likelihood (ML-style) linearization.
}}

\item{max_iter}{Maximum number of EM iterations (outer iterations over \eqn{\beta}
and \eqn{\tau^2}). Can be overridden by \code{control$em_max_iter}.}

\item{tol}{Baseline convergence tolerance for the EM algorithm. The staged thresholds
default to:
\itemize{
\item Laplace stage: \code{tol_laplace = 10 * tol},
\item FE-mean stage: \code{tol_fe_mean = 3 * tol},
\item FE-full stage: \code{tol_fe_full = tol}.
}
You can override these via \code{control$tol_laplace}, \code{control$tol_fe_mean},
and \code{control$tol_fe_full}.}

\item{control}{List of optional control parameters. Recognized entries include:
\itemize{
\item \code{em_max_iter}, \code{em_tol},
\item \code{tol_laplace}, \code{tol_fe_mean}, \code{tol_fe_full},
\item \code{eta_max_iter}, \code{eta_tol_grad},
\item \code{beta_max_iter}, \code{beta_tol},
\item \code{tau2_init} (initial value for \eqn{\tau^2}),
\item \code{vc_eps} (lower bound for \eqn{\tau^2}),
\item \code{max_nq_mem} (memory guard for FE trace intermediates),
\item \code{verbose} (logical),
\item \code{beta_step_max} (max Newton step size for beta; default 2),
\item \code{beta_ls_max_iter} (max line-search halvings; default 12),
\item \code{beta_hess_ridge_init} (initial ridge for Hessian; default 1e-8),
\item \code{beta_hess_ridge_max} (max ridge; default 1e2)
}}
}
\value{
A fitted model object of class \code{glmmFELMod}.
}
\description{
\code{glmmFEL()} fits a generalized linear mixed model (GLMM) with multivariate
normal random effects using EM-type algorithms and likelihood approximations:
\itemize{
\item first-order Laplace (\code{approx = "Laplace"}),
\item fully exponential corrections to the random-effects mean
(\code{approx = "FE_mean"}),
\item fully exponential corrections to both mean and covariance
(\code{approx = "FE_full"} / \code{"FE"}),
\item pseudo-likelihood / PL linearization (\code{approx = "RSPL"} or \code{"MSPL"})
via \code{\link[=glmmFEL_pl]{glmmFEL_pl()}}.
}

This \strong{development branch is matrix-only}: you provide the response \code{y},
fixed-effects design matrix \code{X}, and random-effects design matrix \code{Z}.
A formula interface (via optional 'lme4' helpers) and structured \eqn{G}
parameterizations are in development.

Random effects are assumed \eqn{\eta \sim N(0, G)} with a \strong{single} variance
component
\deqn{G = \tau^2 I_q,}
allowing arbitrary (including multi-membership) \code{Z} while keeping the variance
update simple and stable.
}
\section{NOTE}{

This matrix-only development branch is a streamlined rewrite based on the
binary/Poisson engines in \pkg{mvglmmRank} (which were based on Karl, Yang,
and Lohr, 2014). The fully exponential Laplace EM strategy is described in
Karl, Yang, and Lohr (2014) and related fully exponential Laplace work such
as Tierney, Kass, and Kadane (1989) and Rizopoulos, Verbeke, and Lesaffre (2009);
the FE-within-EM lineage is attributed to Steele (1996). Karl, Yang, and Lohr (2014)
referenced the source code of package JM when writing their own code.

At present, the package supports a single random-effect vector (i.e., one variance
component). However, the theory in the references above—and the joint Poisson-binary
model already implemented in \pkg{mvglmmRank}—extends naturally to a block-diagonal
random-effects covariance matrix \eqn{G}. This would allow multiple independent random
effects and/or random intercept–slope specifications, with intercepts and slopes either
correlated or uncorrelated (depending on the block structure). Enabling this functionality
primarily involves allowing the user to specify a \eqn{G} structure in the main fitting
function. The maintainer has working prototype code for selected \eqn{G} structures, but it
has not yet been tested sufficiently for release; future support may be added. If you would
like a particular \eqn{G} structure supported, please email the package maintainer with a
request.

Importantly, the current implementation already supports a multiple-membership (or otherwise
arbitrary) random-effects design matrix \eqn{Z}, which is more general than what is available
in some other implementations. In particular, the multi-membership setting allows more than
one random effect from the same variance component to be active on the same observation, possibly
with different weight entries per random effect; see Karl, Yang, and Lohr (2014) for details.

The \code{RSPL}/\code{MSPL} pseudo-likelihood code paths are adapted from the
RealVAMS implementation described in Broatch, Green, and Karl (2018), which
follows Wolfinger and O'Connell (1993) closely (working response + working
weights).
}

\section{Acknowledgments}{

OpenAI's GPT models (such as GPT-5 Pro) were
used to assist with coding and roxygen documentation; all
content was reviewed and finalized by the author.
}

\examples{
## Example 1: Simulated probit random-intercept GLMM (matrix interface)
set.seed(1)
n_id <- 30
m_per_id <- 6
n <- n_id * m_per_id
id <- factor(rep(seq_len(n_id), each = m_per_id))
x  <- rnorm(n)
X  <- model.matrix(~ x)
Z  <- Matrix::sparseMatrix(i = seq_len(n),
                           j = as.integer(id),
                           x = 1,
                           dims = c(n, n_id))
beta_true <- c(0.2, 0.7)
tau2_true <- 0.5
eta_true  <- rnorm(n_id, sd = sqrt(tau2_true))
lp <- as.vector(X \%*\% beta_true + Z \%*\% eta_true)
y  <- rbinom(n, 1, pnorm(lp))

fit <- glmmFEL(y, X, Z, family = "binomial_probit", approx = "Laplace")
fit$beta
fit$tau2

## Example 2: Get X, y, Z from an lme4 formula (without a glmmFEL formula wrapper)
\donttest{
if (requireNamespace("lme4", quietly = TRUE)) {
  dat <- data.frame(y = y, x = x, id = id)
  lf  <- lme4::lFormula(y ~ x + (1 | id), data = dat)
  X_lme4 <- lf$X
  Z_lme4 <- Matrix::t(lf$reTrms$Zt)
  y_lme4 <- lf$fr$y

  fit2 <- glmmFEL(y_lme4, X_lme4, Z_lme4, family = "binomial_probit", approx = "Laplace")
}
}

}
\references{
Broatch, J., Green, J. G., & Karl, A. T. (2018).
RealVAMS: An R Package for Fitting a Multivariate Value-added Model (VAM).
\emph{The R Journal}, \emph{10}(1), 22--30.
\doi{10.32614/RJ-2018-033}

Karl, A. T., Yang, Y., & Lohr, S. L. (2014).
Computation of maximum likelihood estimates for multiresponse generalized
linear mixed models with non-nested, correlated random effects.
\emph{Computational Statistics & Data Analysis}, \emph{73}, 146--162.
\doi{10.1016/j.csda.2013.11.019}

Rizopoulos, D., Verbeke, G., & Lesaffre, E. (2009).
Fully exponential Laplace approximations in joint models for longitudinal
and survival data.
\emph{Journal of the Royal Statistical Society: Series B (Statistical Methodology)},
\emph{71}(3), 637--654.
\doi{10.1111/j.1467-9868.2008.00704.x}

Rizopoulos, D. (2010).
JM: An R package for the joint modeling of longitudinal and time-to-event data.
\emph{Journal of Statistical Software}, \emph{35}(9), 1--33.
\doi{10.18637/jss.v035.i09}

Steele, B. M. (1996).
A modified EM algorithm for estimation in generalized mixed models.
\emph{Biometrics}, \emph{52}(4), 1295--1310.
\doi{10.2307/2532845}

Tierney, L., Kass, R. E., & Kadane, J. B. (1989).
Fully exponential Laplace approximations to expectations and variances of nonpositive functions.
\emph{Journal of the American Statistical Association}, \emph{84}(407), 710--716.
\doi{10.1080/01621459.1989.10478824}

Wolfinger, R., & O'Connell, M. (1993).
Generalized linear mixed models: a pseudo-likelihood approach.
\emph{Journal of Statistical Computation and Simulation}, \emph{48}(3--4), 233--243.
\doi{10.1080/00949659308811554}
}
