% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PanelSurvey.R
\name{extract_surveys}
\alias{extract_surveys}
\title{Extract surveys by periodicity from a rotating panel}
\usage{
extract_surveys(
  RotativePanelSurvey,
  index = NULL,
  monthly = NULL,
  annual = NULL,
  quarterly = NULL,
  biannual = NULL,
  use.parallel = FALSE
)
}
\arguments{
\item{RotativePanelSurvey}{A \code{RotativePanelSurvey} object containing
the rotating panel surveys organized temporally}

\item{index}{Integer vector specifying survey indices to extract. If a single
value, returns that survey; if a vector, returns a list}

\item{monthly}{Integer vector specifying which months to extract for
monthly analysis (1-12)}

\item{annual}{Integer vector specifying which years to extract for
annual analysis}

\item{quarterly}{Integer vector specifying which quarters to extract
for quarterly analysis (1-4)}

\item{biannual}{Integer vector specifying which semesters to extract
for biannual analysis (1-2)}

\item{use.parallel}{Logical indicating whether to use parallel processing
for intensive operations. Default FALSE}
}
\value{
A list of \code{Survey} objects matching the specified criteria,
or a single \code{Survey} object if a single index is specified
}
\description{
Extracts subsets of surveys from a RotativePanelSurvey object based on
temporal criteria. Allows obtaining surveys for different types of analysis
(monthly, quarterly, annual) respecting the rotating panel's temporal
structure.
}
\details{
This function is essential for working with rotating panels because:
\itemize{
\item Enables periodicity-based analysis: Extract data for different
types of temporal estimations
\item Preserves temporal structure: Respects temporal relationships
between different panel waves
\item Optimizes memory: Only loads surveys needed for the analysis
\item Facilitates comparisons: Extract specific periods for
comparative analysis
\item Supports parallelization: For operations with large data volumes
}

Extraction criteria are interpreted according to survey frequency:
\itemize{
\item For monthly ECH: monthly=c(1,3,6) extracts January, March and June
\item For annual analysis: annual=1 typically extracts the first available year
\item For quarterly analysis: quarterly=c(1,4) extracts Q1 and Q4
}

If no criteria are specified, the function returns the implantation survey
with a warning.
}
\examples{
\dontrun{
# Load rotating panel
panel_ech <- load_panel_survey(
  path = "ech_panel_2023.dta",
  svy_type = "ech_panel",
  svy_edition = "2023"
)

# Extract specific monthly surveys
ech_q1 <- extract_surveys(
  panel_ech,
  monthly = c(1, 2, 3) # January, February, March
)

# Extract by index
ech_first <- extract_surveys(panel_ech, index = 1)
ech_several <- extract_surveys(panel_ech, index = c(1, 3, 6))

# Quarterly analysis
ech_Q1_Q4 <- extract_surveys(
  panel_ech,
  quarterly = c(1, 4)
)

# Annual analysis (typically all surveys for the year)
ech_annual <- extract_surveys(
  panel_ech,
  annual = 1
)

# With parallel processing for large volumes
ech_full <- extract_surveys(
  panel_ech,
  monthly = 1:12,
  use.parallel = TRUE
)

# Use in workflow
results <- workflow(
  survey = extract_surveys(panel_ech, quarterly = c(1, 2)),
  svymean(~unemployed, na.rm = TRUE),
  estimation_type = "quarterly"
)
}

}
\seealso{
\code{\link{load_panel_survey}} for loading rotating panels
\code{\link{get_implantation}} for obtaining implantation data
\code{\link{get_follow_up}} for obtaining follow-up data
\code{\link{workflow}} for using extracted surveys in analysis

Other panel-surveys: 
\code{\link{PoolSurvey}},
\code{\link{RotativePanelSurvey}},
\code{\link{get_follow_up}()},
\code{\link{get_implantation}()}
}
\concept{panel-surveys}
\keyword{panel-survey}
