% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/neurospace.R
\name{NeuroSpace}
\alias{NeuroSpace}
\title{NeuroSpace: Spatial Reference System for Neuroimaging Data}
\usage{
NeuroSpace(dim, spacing = NULL, origin = NULL, axes = NULL, trans = NULL)
}
\arguments{
\item{dim}{An integer vector specifying the dimensions of the image grid. Must be positive.}

\item{spacing}{A numeric vector specifying the physical size of each voxel (typically in
millimeters). Must be positive. If NULL, defaults to ones.}

\item{origin}{A numeric vector specifying the real-world coordinates of the first voxel.
If NULL, defaults to zeros.}

\item{axes}{An \code{\linkS4class{AxisSet}} object defining the orientation and ordering
of the coordinate axes. If NULL, defaults to standard neurological convention
(Left-Posterior-Inferior for 3D).}

\item{trans}{A transformation matrix mapping voxel indices to world coordinates. If NULL,
constructed from spacing and origin.}
}
\value{
A new \code{\linkS4class{NeuroSpace}} object
}
\description{
The \code{NeuroSpace} class defines the spatial properties and coordinate system of
neuroimaging data. It encapsulates all information needed to map between voxel indices
and real-world coordinates, including dimensions, voxel spacing, origin, axis orientation,
and coordinate transformations.
}
\details{
Spatial Reference System for Neuroimaging Data
}
\section{Coordinate Systems}{

NeuroSpace manages two coordinate systems:
\itemize{
  \item Voxel coordinates: Zero-based indices into the image grid
  \item World coordinates: Real-world coordinates (typically in millimeters)
}

The transformation between these systems is defined by:
\itemize{
  \item Voxel spacing (physical size of voxels)
  \item Origin (world coordinates of first voxel)
  \item Axis orientation (how image axes map to anatomical directions)
}
}

\section{Validation}{

The constructor performs extensive validation:
\itemize{
  \item All dimensions must be positive integers
  \item All spacing values must be positive
  \item Origin and spacing must have matching lengths
  \item Transformation matrix must be invertible
}
}

\examples{
# Create a standard 3D space (64x64x40 voxels, 2mm isotropic)
space_3d <- NeuroSpace(
  dim = c(64L, 64L, 40L),
  spacing = c(2, 2, 2),
  origin = c(-90, -126, -72)
)

# Check properties
dim(space_3d)           # Image dimensions
spacing(space_3d)       # Voxel sizes
origin(space_3d)        # World-space origin

# Create a 2D slice space
space_2d <- NeuroSpace(
  dim = c(128L, 128L),
  spacing = c(1.5, 1.5),
  origin = c(-96, -96)
)

# Convert between coordinate systems
world_coords <- c(0, 0, 0)
vox_idx <- coord_to_index(space_3d, world_coords)
back_to_world <- index_to_coord(space_3d, vox_idx)

}
\references{
For details on neuroimaging coordinate systems:
\itemize{
  \item Brett, M., Johnsrude, I. S., & Owen, A. M. (2002).
    The problem of functional localization in the human brain.
    Nature Reviews Neuroscience, 3(3), 243-249.
  \item Evans, A. C., et al. (1993). 3D statistical neuroanatomical models
    from 305 MRI volumes. Nuclear Science Symposium and Medical Imaging Conference.
}
}
\seealso{
\code{\linkS4class{AxisSet}} for axis orientation specification,
\code{\link{coord_to_index}} for coordinate conversion,
\code{\link{index_to_coord}} for inverse coordinate conversion,
\code{\linkS4class{NeuroObj}} for objects using NeuroSpace
}
