rand.test.cdf1 <- 
  function(x, y, 
           method = c("AD", "CVM", "KS"),
           R = 9999, parallel = FALSE, cl = NULL,
           perm.dist = TRUE){
    # One-Sample Randomization Test for Distribution Equivalence
    # Nathaniel E. Helwig (helwig@umn.edu)
    # last updated: 2026-01-13
    
    
    #########   INITIAL CHECKS   #########
    
    ### check x
    x <- as.matrix(x)
    n <- nrow(x)
    nvar <- ncol(x)
    
    ### check y
    px <- eval(parse(text = paste0("p", y)))
    rx <- eval(parse(text = paste0("r", y)))
    
    ### check method
    methods <- c("AD", "CVM", "KS")
    method <- as.character(method[1])
    method <- pmatch(method, methods)
    if(is.na(method)) stop("Invalid 'method' input.")
    method <- methods[method]
    
    ### check R
    R <- as.integer(R)
    if(R < 1) stop("Input 'R' must be a positive integer.")
    
    ### check parallel
    parallel <- as.logical(parallel[1])
    
    ### check 'cl'
    make.cl <- FALSE
    if(parallel){
      if(is.null(cl)){
        make.cl <- TRUE
        cl <- parallel::makeCluster(2L)
      } else {
        if(!any(class(cl) == "cluster")) stop("Input 'cl' must be an object of class 'cluster'.")
      }
    }
    
    
    #########   DISTRIBUTION TEST   #########
    
    ### univariate or multivariate
    if(nvar == 1L){
      
      ## UNIVARIATE TEST
      
      ## observed test statistic
      Tstat <- Tstat.cdf1(x = x, px = px, method = method)
      
      ## approximate permutation test (given input R)
      nperm <- R + 1L
      permdist <- rep(0, nperm)
      permdist[1] <- Tstat
      
      ## parallel or sequential computation?
      if(parallel){
        permdist[2:nperm] <- parallel::parSapply(cl = cl, X = integer(R), 
                                                 FUN = Tperm.cdf1, 
                                                 n = n, px = px, rx = rx,
                                                 method = method)
      } else {
        permdist[2:nperm] <- sapply(X = integer(R),
                                    FUN = Tperm.cdf1, 
                                    n = n, px = px, rx = rx,
                                    method = method)
      } # end if(parallel)
      
      ## permutation p-value (greater than alternative = only option)
      p.value <- mean(permdist >= Tstat)
      
    } else {
      
      ## MULTIVARIATE TEST
      
      ## observed test statistic
      Tuni <- Tstat.cdf1.mv(x = x, px = px, method = method, combine = FALSE)
      Tstat <- max(Tuni)
      
      ## approximate permutation test (given input R)
      nperm <- R + 1L
      permdist <- rep(0, nperm)
      permdist[1] <- Tstat
      
      ## parallel or sequential computation?
      if(parallel){
        permdist[2:nperm] <- parallel::parSapply(cl = cl, X = integer(R), 
                                                 FUN = Tperm.cdf1.mv, 
                                                 n = n, nvar = nvar,
                                                 px = px, rx = rx,
                                                 method = method)
      } else {
        permdist[2:nperm] <- sapply(X = integer(R),
                                    FUN = Tperm.cdf1.mv, 
                                    n = n, nvar = nvar,
                                    px = px, rx = rx,
                                    method = method)
      } # end if(parallel)
      
      ## permutation p-value
      uni.p.value <- rep(NA, nvar)
      p.value <- mean(permdist >= Tstat)
      for(v in 1:nvar) uni.p.value[v] <- mean(permdist >= Tuni[v])
      
    } # end if(nvar == 1L)
    
    
    #########   RESULTS   #########
    
    ### return results
    if(make.cl) parallel::stopCluster(cl)
    if(!perm.dist) permdist <- NULL
    res <- list(statistic = Tstat, p.value = p.value,
                perm.dist = permdist, method = method,
                R = R)
    if(nvar > 1L) {
      res$univariate <- Tuni
      res$adj.p.values <- uni.p.value
    }
    class(res) <- "rand.test.cdf1"
    return(res)
    
  } # end rand.test.cdf1


### permutation replication (univariate)
Tperm.cdf1 <-
  function(i, n, px, rx, method = "AD"){
    Tstat <- Tstat.cdf1(x = rx(n),
                        px = px,
                        method = method)
    return(Tstat)
  } # end Tperm.cdf1.R


### permutation replication (multivariate)
Tperm.cdf1.mv <-
  function(i, n, nvar, px, rx, method = "AD", combine = TRUE){
    Tstat <- Tstat.cdf1.mv(x = replicate(nvar, rx(n)), px = px, 
                           method = method, combine = combine)
    return(Tstat)
  } # end Tperm.cdf1.mv.R


### test statistic (univariate)
Tstat.cdf1 <- 
  function(x, px, method = "AD"){
    n <- length(x)
    if(method == "AD"){
      x <- sort(x)
      S <- mean( (2*rank(x) - 1) * (log(px(x)) + log(1 - px(x[n:1]))) / n )
      Tstat <- (-1) * (1 + S)
    } else if(method == "CVM"){
      Tstat <- mean(( (2*rank(x) - 1) / (2 * n) - px(x) )^2) + 1 / (12 * n^2)
    } else {
      Fx <- ecdf(x)
      Tstat <- max((Fx(x) - px(x))^2)
    }
    Tstat
  } # end Tstat.cdf1


### test statistic (multivariate)
Tstat.cdf1.mv <- 
  function(x, px, method = "AD", combine = TRUE){
    nvar <- ncol(x)
    Tstat <- rep(0.0, nvar)
    for(j in 1:nvar) {
      Tstat[j] <- Tstat.cdf1(x = x[,j], px = px, method = method)
    }
    if(combine) Tstat <- max(Tstat)
    Tstat
  } # end Tstat.cdf1.mv