% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bubble_grid.R
\name{bubble_grid}
\alias{bubble_grid}
\title{Build a customizable bubble grid chart}
\usage{
bubble_grid(
  data,
  shape = "circles",
  colors = c("#15607A", "#FFFFFF", "#FA8C00"),
  color_ref = NULL,
  color_by = NULL,
  min_value = NULL,
  max_value = NULL,
  opacity = 1,
  bias = 1,
  number_fmt = NULL,
  text_size = NULL,
  text_color = "black",
  text_color_ref = NULL,
  show_text = TRUE,
  brighten_text = TRUE,
  brighten_text_color = "white",
  bold_text = FALSE,
  span = FALSE,
  box_shadow = FALSE,
  tooltip = FALSE,
  animation = "background 1s ease"
)
}
\arguments{
\item{data}{Dataset containing at least one numeric column.}

\item{shape}{The shape of the bubbles.
Options are 'circles' or 'squares'.
Default is 'circles'.}

\item{colors}{A vector of colors to color the bubbles.
Colors should be given in order from low values to high values.
Default colors provided are blue-white-orange: c("#15607A", "#FFFFFF", "#FA8C00").
Can use R's built-in colors or other color packages.}

\item{color_ref}{Optionally assign colors to from another column
by providing the name of the column containing the colors in quotes.
Only one color can be provided per row.
Default is NULL.}

\item{color_by}{Assign colors to a column based on the values of another column.
The column in reference must contain numeric data.
The column in which the colors are being assigned to can be either numerical or character.
Default is NULL.}

\item{min_value}{A value to use as the minimum value for the size of the bubbles.
Default is NULL.}

\item{max_value}{A value to use as the maximum value for the size of the bubbles.
The default maximum value is the maximum value in the column.
Default is NULL.}

\item{opacity}{A value between 0 and 1 that adjusts the opacity in colors.
A value of 0 is fully transparent, a value of 1 is fully opaque.
Default is 1.}

\item{bias}{A positive value that determines the spacing between multiple colors.
A higher value spaces out the colors at the higher end more than a lower number.
Default is 1.}

\item{number_fmt}{Optionally format numbers using formats from the scales package.
Default is NULL.}

\item{text_size}{Numeric value representing the size of the text labels.
Default is NULL.}

\item{text_color}{Assigns text color to values.
Default is black.}

\item{text_color_ref}{Optionally assign text color from another column
by providing the name of the column containing the text colors in quotes.
Only one color can be provided per cell.
Default is NULL.}

\item{show_text}{Logical: show text or hide text.
Default is TRUE.}

\item{brighten_text}{Logical: automatically assign color to text based on background color of cell.
Text within dark-colored backgrounds will turn white, text within light-colored backgrounds will be black.
Default is TRUE.}

\item{brighten_text_color}{Assigns text color to values if values are within a dark-colored backgrounds.
Default is white.}

\item{bold_text}{Logical: bold text.
Default is FALSE.}

\item{span}{Optionally apply colors to values across multiple columns instead of by each column.
To apply across all columns set to TRUE.
If applying to a set of columns, can provide either column names or column positions.
Default is FALSE.}

\item{box_shadow}{Logical: add a box shadow to the tiles.
Default is FALSE.}

\item{tooltip}{Logical: hover tooltip.
Default is FALSE.}

\item{animation}{Control the duration and timing function of the animation
when sorting/updating values shown on a page.
See [CSS transitions](https://developer.mozilla.org/en-US/docs/Web/CSS/transition)
for available timing functions and examples.
Animation can be turned off by setting to "none".
Animation can be applied to the size of the bubbles by setting it to "all 1s ease".
Default is "background 1s ease".}
}
\value{
a function that builds a bubble grid chart
    to a column of values.
}
\description{
The `bubble_grid()` function creates a customizable bubble grid chart within a reactable table.
    The size of the bubbles are in relation to the values within each column - the bigger the value, the bigger the size of the bubble.
    There are two shapes available for the bubble grid: circles and squares, which can be specified with `shape`.
    The colors can be provided within a vector in `colors` or via another column in the dataset by referencing the column by name with `color_ref`.
    If more than one color is provided in `colors`, the colors will be assigned to the values from low to high within the palette.
    This is the default setting of `bubble_grid()`, which applies a blue-to-orange color palette to the bubbles. However, a singular color can be provided instead if desired.
    `bubble_grid()` can be applied to columns containing character data by referencing another column with numeric values in it with `color_by`.
    The opacity of the colors provided can be adjusted by providing a value between 0 and 1 in `opacity`.
    `text_color` can be used to change the color of the values displayed within the bubbles.
    If values are displayed within a dark-colored background, `brighten_text` will display the values in white text so they are more visible.
    For smaller values with a dark-colored background, the values may not be visible.
    If you would like these numbers to be visible, you could do so by either:
    A) setting `brighten_text` to FALSE and assigning a universal color to the text within `text_color`.
    B) leaving `brighten_text` as TRUE and setting `brighten_text_color` to a darker color other than the default white color.
    If the user wants to assign colors row-wise instead of column-wise, set `span` equal to TRUE to apply across all columns.
    Or can provide the names of the columns by either column name or column position number to apply to only a subset of the columns.
    The format of the numbers within the bubbles can be changed by defining the format from a package such as the {scales} package within `number_fmt`.
    `bubble_grid()` needs to placed within the cell argument in reactable::colDef.
}
\examples{
data <- iris[10:29, ]

## By default, the bubble_grid() function uses a blue-white-orange three-color pattern:
reactable(
  data,
  columns = list(
    Petal.Length = colDef(
      align = "center",
      cell = bubble_grid(data))))

## You can specify your own color palette or a single color across all values with `colors`;
reactable(
  data,
  columns = list(
    Petal.Length = colDef(
      align = "center",
      cell = bubble_grid(data,
                         colors = c("orange")))))

## Use squares instead of circles:
reactable(
  data,
  columns = list(
    Petal.Length = colDef(
      align = "center",
      cell = bubble_grid(data,
                         shape = "squares"))))

## Hide text and show on hover by enabling the tooltip:
reactable(
  data,
  columns = list(
    Petal.Length = colDef(
      align = "center",
      cell = bubble_grid(data,
                         show_text = FALSE,
                         tooltip = TRUE))))

## Control the scale of the circles by adjusting the min and max values:
reactable(
  data,
  columns = list(
    Petal.Length = colDef(
      align = "center",
      cell = bubble_grid(data,
                         min_value = 1,
                         max_value = 2))))

## Use span to apply bubbles to values in relation to the entire data set:
reactable(
  data,
  defaultColDef = colDef(
    cell = bubble_grid(data,
                       span = TRUE)))

## Use number_fmt to format numbers using the scales package:
car_prices <- MASS::Cars93[20:49, c("Make", "Price")]

reactable(car_prices,
  defaultColDef = colDef(
    align = "center",
    cell = bubble_grid(car_prices,
                       number_fmt = scales::dollar)))
}
