#' Get ABARES' "Australian Gridded Farm Data" (AGFD)
#'
#' Used by the `read_agfd` family of functions, downloads the "Australian
#'  Gridded Farm Data" (\acronym{AGFD}) data.
#'
#' @param .fixed_prices Download historical climate and prices or historical
#'  climate and fixed prices as described in (Hughes *et al.* 2022).
#' @param .yyyy Returns only data for the specified year or years for climate
#'  data (fixed prices) or the years for historical climate and prices depending
#'  upon the setting of `.fixed_prices`.  Note that this will still download the
#'  entire data set, that cannot be avoided, but will only return the
#'  requested year(s) in your \R session.  Valid years are from 1991 to 2023
#'  inclusive.
#'
#' @examples
#' # this will download the data and then return only 2020 and 2021 years' data
#' agfd <- .get_agfd(.fixed_prices = TRUE, .yyyy = 2020:2021)
#'
#' agfd
#'
#' @returns A vector object, a list of NetCDF file paths containing the
#' "Australian Gridded Farm Data" data.
#' @autoglobal
#' @dev

.get_agfd <- function(.fixed_prices, .yyyy) {
  .x <- fs::path_temp(
    sprintf(
      "%s.zip",
      data.table::fifelse(
        .fixed_prices,
        "historical_climate_prices_fixed",
        "historical_climate_prices"
      )
    )
  )

  if (!.file_exists(.x)) {
    file_url <- data.table::fifelse(
      .fixed_prices,
      "https://daff.ent.sirsidynix.net.au/client/en_AU/search/asset/1036161/3",
      "https://daff.ent.sirsidynix.net.au/client/en_AU/search/asset/1036161/2"
    )
    .retry_download(
      url = file_url,
      dest = .x
    )
  }

  agfd_nc <- .read_ncdf_from_zip(zip_path = .x, .fixed_prices = .fixed_prices)

  .yyyy <- as.character(.yyyy)
  yyyy <- sprintf("c%s", .yyyy)
  nm <- fs::path_file(agfd_nc)
  agfd_nc <- agfd_nc[grepl(paste(yyyy, collapse = "|"), nm)]
  names(agfd_nc) <- sprintf("year_%s", .yyyy)

  return(agfd_nc)
}

#' Unzip AGFD NetCDF files from ZIP
#'
#' @param zip_path Path to the ZIP file containing NetCDF files.
#' @returns A vector of paths to the extracted NetCDF files.
#' @dev
#'

.read_ncdf_from_zip <- function(
  zip_path = fs::path_temp(),
  .fixed_prices = TRUE
) {
  # Extract only NetCDF files
  utils::unzip(zip_path, exdir = fs::path_temp())

  f <- data.table::fifelse(
    .fixed_prices,
    fs::path_temp("historical_climate_prices_fixed"),
    fs::path_temp("historical_climate_and_prices")
  )

  return(
    unlist(purrr::map(.x = f, .f = fs::dir_ls))
  )
}

#' Copies local AGFD file to tempdir() so the org file isn't touched
#' @param x A filepath pointing to the AGFD file provided by the user
#' @returns A character vector of file paths for AGFD netCDF files
#' @dev
.copy_local_agfd_zip <- function(x) {
  y <- fs::path_temp(fs::path_file(x))
  fs::file_copy(x, y, overwrite = TRUE)
  return(.read_ncdf_from_zip(zip_path = y))
}

#' Internal wrapper for fs::file_exists
#'
#' @dev
.file_exists <- function(path) {
  fs::file_exists(path)
}
