% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semmcci-mc-generic.R
\name{MCGeneric}
\alias{MCGeneric}
\title{Monte Carlo Confidence Intervals (Generic)}
\usage{
MCGeneric(
  object,
  def,
  R = 20000L,
  alpha = c(0.001, 0.01, 0.05),
  decomposition = "eigen",
  pd = TRUE,
  tol = 1e-06,
  seed = NULL
)
}
\arguments{
\item{object}{R object.
Fitted model object with \code{coef} and \code{vcov} methods
that return a named vector of
estimated parameters and sampling variance-covariance matrix,
respectively.}

\item{def}{List of character strings.
A list of defined functions of parameters.
The string should be a valid R expression when parsed
and should result a single value when evaluated.}

\item{R}{Positive integer.
Number of Monte Carlo replications.}

\item{alpha}{Numeric vector.
Significance level \eqn{\alpha}.}

\item{decomposition}{Character string.
Matrix decomposition of the sampling variance-covariance matrix
for the data generation.
If \code{decomposition = "chol"}, use Cholesky decomposition.
If \code{decomposition = "eigen"}, use eigenvalue decomposition.
If \code{decomposition = "svd"}, use singular value decomposition.}

\item{pd}{Logical.
If \code{pd = TRUE},
check if the sampling variance-covariance matrix
is positive definite using \code{tol}.}

\item{tol}{Numeric.
Tolerance used for \code{pd}.}

\item{seed}{Integer.
Random seed for reproducibility.}
}
\value{
Returns an object of class \code{semmcci} which is
a list with the following elements:
\describe{
\item{call}{Function call.}
\item{args}{List of function arguments.}
\item{thetahat}{Parameter estimates \eqn{\hat{\theta}}.}
\item{thetahatstar}{Sampling distribution of parameter estimates
\eqn{\hat{\theta}^{\ast}}.}
\item{fun}{Function used ("MCGeneric").}
}
}
\description{
Calculates Monte Carlo confidence intervals
for defined parameters
for any fitted model object with \code{coef} and \code{vcov} methods.
}
\details{
A sampling distribution of parameter estimates is generated
from the multivariate normal distribution
using the parameter estimates and the sampling variance-covariance matrix.
Confidence intervals for defined parameters
are generated using the simulated sampling distribution.
Parameters are defined using the \code{def} argument.
}
\examples{
library(semmcci)
library(lavaan)

# Data ---------------------------------------------------------------------
data("Tal.Or", package = "psych")
df <- mice::ampute(Tal.Or)$amp

# Monte Carlo --------------------------------------------------------------
## Fit Model in lavaan -----------------------------------------------------
model <- "
  reaction ~ cp * cond + b * pmi
  pmi ~ a * cond
  cond ~~ cond
"
fit <- sem(data = df, model = model, missing = "fiml")

## MCGeneric() -------------------------------------------------------------
MCGeneric(
  fit,
  R = 5L, # use a large value e.g., 20000L for actual research
  alpha = 0.05,
  def = list(
    "a * b",
    "cp + (a * b)"
  )
)

}
\references{
MacKinnon, D. P., Lockwood, C. M., & Williams, J. (2004).
Confidence limits for the indirect effect:
Distribution of the product and resampling methods.
\emph{Multivariate Behavioral Research}, \emph{39}(1), 99-128.
\doi{10.1207/s15327906mbr3901_4}

Pesigan, I. J. A., & Cheung, S. F. (2024).
Monte Carlo confidence intervals for the indirect effect with missing data.
\emph{Behavior Research Methods}.
\doi{10.3758/s13428-023-02114-4}

Preacher, K. J., & Selig, J. P. (2012).
Advantages of Monte Carlo confidence intervals for indirect effects.
\emph{Communication Methods and Measures}, \emph{6}(2), 77–98.
\doi{10.1080/19312458.2012.679848}
}
\seealso{
Other Monte Carlo in Structural Equation Modeling Functions: 
\code{\link{Func}()},
\code{\link{MC}()},
\code{\link{MCFunc}()},
\code{\link{MCMI}()},
\code{\link{MCStd}()}
}
\author{
Ivan Jacob Agaloos Pesigan
}
\concept{Monte Carlo in Structural Equation Modeling Functions}
\keyword{mc}
\keyword{semmcci}
