% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PostProcess_TreeLocations.R
\name{process_tree_data}
\alias{process_tree_data}
\title{Obtain tree information by processing point cloud data}
\usage{
process_tree_data(treeData, segmentedLAS, return_sf = FALSE)
}
\arguments{
\item{treeData}{An \code{sf} object containing the following tree information: \code{TreeID},
\code{X}, \code{Y}, \code{Z}, \code{Radius}, and \code{Error}, output from the \code{get_raster_eigen_treelocs} function.}

\item{segmentedLAS}{A LAS object that is the output from \code{segment_graph}.}

\item{return_sf}{logical: If TRUE, returns an \code{sf} object representing the convex hulls for each tree.}
}
\value{
sf object An updated \code{sf} object with the original columns plus:
\describe{
\item{height}{numeric: Height of the highest point for each TreeID.}
\item{crown_area}{numeric: Area of the convex hull for each TreeID.}
\item{crown_base_height}{numeric: Height to the base of the live crown for each TreeID.}
\item{crown_volume}{numeric: Volume of the convex hull for the crown of each TreeID.}
\item{diameter}{numeric: Diameter of the tree, calculated as twice the Radius.}
}
If \code{return_sf} is TRUE, returns an \code{sf} object where the geometry is the convex hulls for each tree.
If \code{return_sf} is FALSE, returns an \code{sf} object with point geometries using treeData.
}
\description{
\code{process_tree_data} processes the output of \code{get_raster_eigen_treelocs} and \code{segment_graph} to add information
about the height, crown area, and diameter for each unique TreeID. It also has an optional parameter to return
an \code{sf} object representing the convex hulls for each tree.
}
\details{
For terrestrial and mobile lidar datasets, tree locations and estimates of DBH are provided
by rasterizing individual point cloud values of relative neighborhood density (at 0.3 and
1 m radius) and verticality within a slice of the normalized point cloud around breast height
(1.34 m). The algorithm then uses defined threshold values to classify the resulting rasters
and create unique polygons from the resulting classified raster. These point-density and
verticality polygons were selected by their intersection with one another, resulting in a
final set of polygons which were used to clip out regions of the point cloud that were most
likely to represent tree boles. A RANSAC cylinder fitting algorithm was then used to estimate
the fit of a cylinder to individual bole points. Cylinder centers and radius were used as inputs
to an individual tree segmentation.
}
\examples{

\donttest{
# Set the number of threads to use in lidR
set_lidr_threads(8)

LASfile = system.file("extdata", "TLS_Clip.laz", package="spanner")
las = readTLSLAS(LASfile, select = "xyzcr", "-filter_with_voxel 0.01")
# Don't forget to make sure the las object has a projection
sf::st_crs(las) <- 26912

# Pre-process the example lidar dataset by classifying the ground  and noise points
# using lidR::csf(), normalizing it, and removing outlier points
# using lidR::ivf()
# las = classify_ground(las, csf(sloop_smooth = FALSE,
#                                 class_threshold = 0.5,
#                                cloth_resolution = 0.5, rigidness = 1L,
#                                 iterations = 500L, time_step = 0.65))
# las = normalize_height(las, tin())
# las = classify_noise(las, ivf(0.25, 3))
# las = filter_poi(las, Classification != LASNOISE)

# Plot the non-ground points, colored by height
# plot(filter_poi(las, Classification != 2), color = "Z")

# Find individual tree locations and attribute data
# find tree locations and attribute data
myTreeLocs = get_raster_eigen_treelocs(las = las, res = 0.025, pt_spacing = 0.0254,
                                       dens_threshold = 0.25,
                                       neigh_sizes = c(0.25, 0.15, 0.66),
                                       eigen_threshold = 0.75,
                                       grid_slice_min = 1,
                                       grid_slice_max = 2,
                                       minimum_polygon_area = 0.005,
                                       cylinder_fit_type = "ransac",
                                       max_dia = 1,
                                       SDvert = 0.33,
                                       n_pts = 20,
                                       n_best = 25,
                                       inliers = 0.9,
                                       conf = 0.99,
                                       max_angle = 20)

# Plot results if trees were found
if (!is.null(myTreeLocs) && nrow(myTreeLocs) > 0) {
  plot(lidR::rasterize_canopy(las, res = 0.2, p2r()))
  symbols(sf::st_coordinates(myTreeLocs)[,1], sf::st_coordinates(myTreeLocs)[,2],
          circles = myTreeLocs$Radius^2*3.14, inches = FALSE, add = TRUE, bg = 'black')
} else {
  message("No tree locations were found. Try adjusting the parameters.")
}

# Segment the point cloud
# For areas with interlocking crowns and trees of different sizes,
# enable metabolic scaling to prevent height overestimation
myTreeGraph = segment_graph(las = las, tree.locations = myTreeLocs, k = 50,
                             distance.threshold = 0.5,
                             use.metabolic.scale = FALSE,
                             ptcloud_slice_min = 1,
                             ptcloud_slice_max = 2,
                             subsample.graph = 0.1,
                             return.dense = TRUE)

# Plot it in 3D colored by treeID
plot(myTreeGraph, color = "treeID", pal=spanner_pal())

# Process the data
processed_data <- process_tree_data(myTreeLocs, myTreeGraph, return_sf = TRUE)

# Print the processed data
print(processed_data$data)
# Print the sf object if return_sf is TRUE
if (!is.null(processed_data$sf)) {
  print(processed_data$sf)
}
}

}
