% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get-compustat.R
\name{get_compustat}
\alias{get_compustat}
\title{Download Compustat fundamentals}
\usage{
get_compustat(
  wrds,
  frequency = c("annual", "quarterly"),
  region = c("na", "global"),
  start_date = NULL,
  end_date = NULL,
  columns = NULL,
  add_columns = NULL,
  indfmt = "INDL",
  consol = "C",
  fill_sic = FALSE,
  n = Inf,
  lazy = FALSE
)
}
\arguments{
\item{wrds}{A \code{DBIConnection} object returned by \code{\link[=wrds_connect]{wrds_connect()}}.}

\item{frequency}{One of \code{"annual"} (default) or \code{"quarterly"}.}

\item{region}{One of \code{"na"} (North America, default) or \code{"global"}.}

\item{start_date}{Start date for filtering. Character string in
\code{"YYYY-MM-DD"} format or a Date object. Defaults to \code{NULL} (no filter).}

\item{end_date}{End date for filtering. Character string in
\code{"YYYY-MM-DD"} format or a Date object. Defaults to \code{NULL} (no filter).}

\item{columns}{Character vector of columns to return, replacing the defaults.
Use \code{\link[=describe_table]{describe_table()}} to see available columns.}

\item{add_columns}{Character vector of additional columns to include beyond
the defaults. Ignored if \code{columns} is specified.}

\item{indfmt}{Industry format filter. Defaults to \code{"INDL"} (industrial).
Use \code{"FS"} for financial services format.}

\item{consol}{Consolidation level. Defaults to \code{"C"} (consolidated).
Use \code{"B"} for both consolidated and non-consolidated.}

\item{fill_sic}{If \code{TRUE}, fills missing historical SIC codes (\code{sich}) with
header SIC codes from \code{comp.company}. Only supported for North America.
When used with \code{lazy = TRUE}, returns the table with \code{sich} but without
the join (requires manual joining with \code{\link[=get_company]{get_company()}}). Defaults to
\code{FALSE}.}

\item{n}{Maximum number of rows to return. Defaults to \code{Inf} (all rows).
Use a smaller value (e.g., \code{n = 100}) to preview data before downloading
the full table.}

\item{lazy}{If \code{TRUE}, returns a lazy \code{tbl} instead of collecting.
Defaults to \code{FALSE}.}
}
\value{
A tibble with Compustat fundamentals. Default columns vary by region:

\strong{North America} (from \code{comp.funda} / \code{comp.fundq}):
\itemize{
\item Identifiers: \code{gvkey}, \code{cusip}, \code{tic}, \code{conm}, \code{datadate}
\item Time: \code{fyear}/\code{fyearq}, \code{fyr}/\code{fqtr}
\item Income: \code{ni}/\code{niq}, \code{ib}/\code{ibq}, \code{oiadp}/\code{oiadpq}, \code{revt}/\code{revtq}
\item Balance sheet: \code{at}/\code{atq}, \code{lt}/\code{ltq}, \code{seq}/\code{seqq}, \code{ceq}/\code{ceqq}
\item Market: \code{csho}/\code{cshoq}, \code{prcc_f}/\code{prccq}
\item Other: \code{sale}/\code{saleq}, \code{capx}/\code{capxy}, \code{che}/\code{cheq}, \code{dlc}/\code{dlcq}, \code{dltt}/\code{dlttq}
\item Industry: \code{sich} (historical SIC); \code{sic} (when \code{fill_sic = TRUE}, coalesced
from \code{sich} and header SIC)
}

\strong{Global} (from \code{comp.g_funda} / \code{comp.g_fundq}):
\itemize{
\item Identifiers: \code{gvkey}, \code{isin}, \code{conm}, \code{datadate}
\item Geography: \code{loc}, \code{fic}, \code{exchg}
\item Similar financial variables (with some differences, e.g., \code{nit}/\code{nitq}
instead of \code{ni}/\code{niq})
}
}
\description{
Downloads financial statement data from Compustat with standard filters
for clean, analysis-ready data.
}
\details{
Default filters follow standard practice for most research applications.
Region-specific filters are applied automatically based on \code{region}:
\itemize{
\item \code{datafmt}: \code{"STD"} for North America, \code{"HIST_STD"} for Global
\item \code{popsrc}: \code{"D"} (domestic) for North America, \code{"I"} (international) for Global
}

North America and Global data have different structures and should not
be combined without careful column harmonization.
}
\examples{
\dontrun{
wrds <- wrds_connect()

# Annual North America fundamentals
funda <- get_compustat(wrds)

# Quarterly with date filter
fundq <- get_compustat(wrds,
  frequency = "quarterly",
  start_date = "2020-01-01",
  end_date = "2023-12-31"
)

# Global annual
g_funda <- get_compustat(wrds, region = "global")

# Lazy query for further filtering
get_compustat(wrds, lazy = TRUE) |>
  dplyr::filter(fyear >= 2020) |>
  dplyr::select(gvkey, datadate, at, lt) |>
  dplyr::collect()

# Fill missing SIC codes with header SIC from comp.company
funda_sic <- get_compustat(wrds, fill_sic = TRUE)

# Preview first 100 rows before full download
preview <- get_compustat(wrds, n = 100)

wrds_disconnect(wrds)
}
}
\seealso{
\code{\link[=link_ccm]{link_ccm()}} for CRSP-Compustat linking, \code{\link[=get_company]{get_company()}} for
company header data
}
