use crate::{Axis, Dimension, Ixs};


pub(crate) fn axes_of<'a, D>(d: &'a D, strides: &'a D) -> Axes<'a, D>
where D: Dimension
{
    Axes {
        dim: d,
        strides,
        start: 0,
        end: d.ndim(),
    }
}








/// # Examples
















#[derive(Debug)]
pub struct Axes<'a, D>
{
    dim: &'a D,
    strides: &'a D,
    start: usize,
    end: usize,
}


#[derive(Debug)]
pub struct AxisDescription
{
    
    pub axis: Axis,
    
    pub len: usize,
    
    pub stride: isize,
}

copy_and_clone!(AxisDescription);
copy_and_clone!(['a, D] Axes<'a, D>);

impl<'a, D> Iterator for Axes<'a, D>
where D: Dimension
{
    
    type Item = AxisDescription;

    fn next(&mut self) -> Option<Self::Item>
    {
        if self.start < self.end {
            let i = self.start.post_inc();
            Some(AxisDescription {
                axis: Axis(i),
                len: self.dim[i],
                stride: self.strides[i] as Ixs,
            })
        } else {
            None
        }
    }

    fn fold<B, F>(self, init: B, f: F) -> B
    where F: FnMut(B, AxisDescription) -> B
    {
        (self.start..self.end)
            .map(move |i| AxisDescription {
                axis: Axis(i),
                len: self.dim[i],
                stride: self.strides[i] as isize,
            })
            .fold(init, f)
    }

    fn size_hint(&self) -> (usize, Option<usize>)
    {
        let len = self.end - self.start;
        (len, Some(len))
    }
}

impl<'a, D> DoubleEndedIterator for Axes<'a, D>
where D: Dimension
{
    fn next_back(&mut self) -> Option<Self::Item>
    {
        if self.start < self.end {
            let i = self.end.pre_dec();
            Some(AxisDescription {
                axis: Axis(i),
                len: self.dim[i],
                stride: self.strides[i] as Ixs,
            })
        } else {
            None
        }
    }
}

trait IncOps: Copy
{
    fn post_inc(&mut self) -> Self;
    fn pre_dec(&mut self) -> Self;
}

impl IncOps for usize
{
    #[inline(always)]
    fn post_inc(&mut self) -> Self
    {
        let x = *self;
        *self += 1;
        x
    }
    #[inline(always)]
    fn pre_dec(&mut self) -> Self
    {
        *self -= 1;
        *self
    }
}
