use super::{
    arch::*,
    data::{Map, Stat, StatVfs, TimeSpec},
    error::Result,
    flag::*,
    number::*,
};

use core::mem;


pub fn close(fd: usize) -> Result<usize> {
    unsafe { syscall1(SYS_CLOSE, fd) }
}


pub fn clock_gettime(clock: usize, tp: &mut TimeSpec) -> Result<usize> {
    unsafe { syscall2(SYS_CLOCK_GETTIME, clock, tp as *mut TimeSpec as usize) }
}


pub fn dup(fd: usize, buf: &[u8]) -> Result<usize> {
    unsafe { syscall3(SYS_DUP, fd, buf.as_ptr() as usize, buf.len()) }
}


pub fn dup2(fd: usize, newfd: usize, buf: &[u8]) -> Result<usize> {
    unsafe { syscall4(SYS_DUP2, fd, newfd, buf.as_ptr() as usize, buf.len()) }
}


pub fn exit(status: usize) -> Result<usize> {
    unsafe { syscall1(SYS_EXIT, status) }
}


pub fn fchmod(fd: usize, mode: u16) -> Result<usize> {
    unsafe { syscall2(SYS_FCHMOD, fd, mode as usize) }
}


pub fn fchown(fd: usize, uid: u32, gid: u32) -> Result<usize> {
    unsafe { syscall3(SYS_FCHOWN, fd, uid as usize, gid as usize) }
}


pub fn fcntl(fd: usize, cmd: usize, arg: usize) -> Result<usize> {
    unsafe { syscall3(SYS_FCNTL, fd, cmd, arg) }
}




/// # Errors






pub unsafe fn fmap(fd: usize, map: &Map) -> Result<usize> {
    syscall3(
        SYS_FMAP,
        fd,
        map as *const Map as usize,
        mem::size_of::<Map>(),
    )
}


pub unsafe fn funmap(addr: usize, len: usize) -> Result<usize> {
    syscall2(SYS_FUNMAP, addr, len)
}


pub fn fpath(fd: usize, buf: &mut [u8]) -> Result<usize> {
    unsafe { syscall3(SYS_FPATH, fd, buf.as_mut_ptr() as usize, buf.len()) }
}


pub fn frename<T: AsRef<str>>(fd: usize, path: T) -> Result<usize> {
    let path = path.as_ref();
    unsafe {
        syscall3(
            SYS_FRENAME,
            fd,
            path.as_ptr() as usize,
            path.len(),
        )
    }
}


pub fn fstat(fd: usize, stat: &mut Stat) -> Result<usize> {
    unsafe {
        syscall3(
            SYS_FSTAT,
            fd,
            stat as *mut Stat as usize,
            mem::size_of::<Stat>(),
        )
    }
}


pub fn fstatvfs(fd: usize, stat: &mut StatVfs) -> Result<usize> {
    unsafe {
        syscall3(
            SYS_FSTATVFS,
            fd,
            stat as *mut StatVfs as usize,
            mem::size_of::<StatVfs>(),
        )
    }
}


pub fn fsync(fd: usize) -> Result<usize> {
    unsafe { syscall1(SYS_FSYNC, fd) }
}


pub fn ftruncate(fd: usize, len: usize) -> Result<usize> {
    unsafe { syscall2(SYS_FTRUNCATE, fd, len) }
}


pub fn futimens(fd: usize, times: &[TimeSpec]) -> Result<usize> {
    unsafe {
        syscall3(
            SYS_FUTIMENS,
            fd,
            times.as_ptr() as usize,
            times.len() * mem::size_of::<TimeSpec>(),
        )
    }
}


pub unsafe fn futex(
    addr: *mut i32,
    op: usize,
    val: i32,
    val2: usize,
    addr2: *mut i32,
) -> Result<usize> {
    syscall5(
        SYS_FUTEX,
        addr as usize,
        op,
        (val as isize) as usize,
        val2,
        addr2 as usize,
    )
}


pub fn getegid() -> Result<usize> {
    unsafe { syscall0(SYS_GETEGID) }
}


pub fn getens() -> Result<usize> {
    unsafe { syscall0(SYS_GETENS) }
}


pub fn geteuid() -> Result<usize> {
    unsafe { syscall0(SYS_GETEUID) }
}


pub fn getgid() -> Result<usize> {
    unsafe { syscall0(SYS_GETGID) }
}


pub fn getns() -> Result<usize> {
    unsafe { syscall0(SYS_GETNS) }
}


pub fn getpid() -> Result<usize> {
    unsafe { syscall0(SYS_GETPID) }
}


pub fn getpgid(pid: usize) -> Result<usize> {
    unsafe { syscall1(SYS_GETPGID, pid) }
}


pub fn getppid() -> Result<usize> {
    unsafe { syscall0(SYS_GETPPID) }
}


pub fn getuid() -> Result<usize> {
    unsafe { syscall0(SYS_GETUID) }
}



/// # Errors



pub unsafe fn iopl(level: usize) -> Result<usize> {
    syscall1(SYS_IOPL, level)
}


pub fn kill(pid: usize, sig: usize) -> Result<usize> {
    unsafe { syscall2(SYS_KILL, pid, sig) }
}


pub unsafe fn link(old: *const u8, new: *const u8) -> Result<usize> {
    syscall2(SYS_LINK, old as usize, new as usize)
}


pub fn lseek(fd: usize, offset: isize, whence: usize) -> Result<usize> {
    unsafe { syscall3(SYS_LSEEK, fd, offset as usize, whence) }
}


pub fn mkns(schemes: &[[usize; 2]]) -> Result<usize> {
    unsafe { syscall2(SYS_MKNS, schemes.as_ptr() as usize, schemes.len()) }
}


pub unsafe fn mprotect(addr: usize, size: usize, flags: MapFlags) -> Result<usize> {
    syscall3(SYS_MPROTECT, addr, size, flags.bits())
}


pub fn nanosleep(req: &TimeSpec, rem: &mut TimeSpec) -> Result<usize> {
    unsafe {
        syscall2(
            SYS_NANOSLEEP,
            req as *const TimeSpec as usize,
            rem as *mut TimeSpec as usize,
        )
    }
}


pub fn open<T: AsRef<str>>(path: T, flags: usize) -> Result<usize> {
    let path = path.as_ref();
    unsafe {
        syscall3(
            SYS_OPEN,
            path.as_ptr() as usize,
            path.len(),
            flags,
        )
    }
}


pub fn openat<T: AsRef<str>>(fd: usize, path: T, flags: usize) -> Result<usize> {
    let path = path.as_ref();
    unsafe {
        syscall4(
            SYS_OPENAT,
            fd,
            path.as_ptr() as usize,
            path.len(),
            flags,
        )
    }
}


pub fn read(fd: usize, buf: &mut [u8]) -> Result<usize> {
    unsafe { syscall3(SYS_READ, fd, buf.as_mut_ptr() as usize, buf.len()) }
}


pub fn rmdir<T: AsRef<str>>(path: T) -> Result<usize> {
    let path = path.as_ref();
    unsafe {
        syscall2(
            SYS_RMDIR,
            path.as_ptr() as usize,
            path.len(),
        )
    }
}


pub fn setpgid(pid: usize, pgid: usize) -> Result<usize> {
    unsafe { syscall2(SYS_SETPGID, pid, pgid) }
}


pub fn setregid(rgid: usize, egid: usize) -> Result<usize> {
    unsafe { syscall2(SYS_SETREGID, rgid, egid) }
}


pub fn setrens(rns: usize, ens: usize) -> Result<usize> {
    unsafe { syscall2(SYS_SETRENS, rns, ens) }
}


pub fn setreuid(ruid: usize, euid: usize) -> Result<usize> {
    unsafe { syscall2(SYS_SETREUID, ruid, euid) }
}


pub fn unlink<T: AsRef<str>>(path: T) -> Result<usize> {
    let path = path.as_ref();
    unsafe {
        syscall2(
            SYS_UNLINK,
            path.as_ptr() as usize,
            path.len(),
        )
    }
}



/// # Errors


pub unsafe fn virttophys(virtual_address: usize) -> Result<usize> {
    syscall1(SYS_VIRTTOPHYS, virtual_address)
}


pub fn waitpid(pid: usize, status: &mut usize, options: WaitFlags) -> Result<usize> {
    unsafe {
        syscall3(
            SYS_WAITPID,
            pid,
            status as *mut usize as usize,
            options.bits(),
        )
    }
}







/// # Errors







pub fn write(fd: usize, buf: &[u8]) -> Result<usize> {
    unsafe { syscall3(SYS_WRITE, fd, buf.as_ptr() as usize, buf.len()) }
}




pub fn sched_yield() -> Result<usize> {
    unsafe { syscall0(SYS_YIELD) }
}





pub fn sendfd(receiver_socket: usize, fd: usize, flags: usize, arg: u64) -> Result<usize> {
    #[cfg(target_pointer_width = "32")]
    unsafe {
        syscall5(
            SYS_SENDFD,
            receiver_socket,
            fd,
            flags,
            arg as u32 as usize,
            (arg >> 32) as u32 as usize,
        )
    }

    #[cfg(target_pointer_width = "64")]
    unsafe {
        syscall4(SYS_SENDFD, receiver_socket, fd, flags, arg as usize)
    }
}
