use core:lang;
use core:asm;
use lang:bs:macro;

/**
 * State used when expanding declarations inside a type.
 */
class TypeState on Compiler {
	init(CppType to, Visibility start) {
		init() { to = to; visibility = start; }
	}

	// Add to.
	CppType to;

	// Current visibility.
	Visibility visibility;

	// Set access.
	void set(Visibility v) {
		visibility = v;
	}

	// Add a member.
	void add(Named named) {
		named.visibility = visibility;
		to.add(named);
	}

	// Add a function decl.
	void add(CppDecl decl) {
		decl.visibility = visibility;
		decl.create(to);
	}
}


/**
 * Type declaration (both structs and classes, we don't support unions).
 */
class CppType extends Type {
	// If 'isClass' is 'false', then we're a struct, i.e. access defaults to public rather than private.
	init(SStr name, Bool isClass, SClassInherit? inherit, SClassBody body) {
		init(name.pos, name.v, TypeFlags:typeValue) {
			inherit = inherit;
			body = body;
			defAccess = if (isClass) { typePrivate; } else { public; };
		}
	}

	// Body.
	SClassBody body;

	// Inheritance, expanded when we're told to.
	SClassInherit? inherit;

	// Default access.
	Visibility defAccess;

	// Resolve types etc. Called when all types in this package are loaded.
	void resolve() {
		if (inherit)
			inherit.transform(this);
	}

	// Get a scope.
	Scope scope() {
		cppScope(this);
	}

	// Called by the grammar to set up inheritance.
	void setSuper(Visibility? vis, Value type) {
		// Currently, we don't really care about the visibility, as it is not supported in the backend.
		setSuper(type.type);
	}

	// Called to load all content.
	Bool loadAll() : override {
		TypeState state(this, defAccess);
		body.transform(state);

		// Check for special member functions.
		Bool hasCtor = false;
		Bool hasCopyCtor = false;
		Bool hasAssignFn = false;
		Bool hasDtor = false;
		for (member in this) {
			if (member.name == "__destroy")
				hasDtor = true;
			if (member.name == "__init") {
				if (isCopyFn(this, member))
					hasCopyCtor = true;
				else
					hasCtor = true;
			}
			if (member.name == "=") {
				if (isCopyFn(this, member))
					hasAssignFn = true;
			}
		}

		// Add a default constructor if one was not declared.
		if (!hasCtor) {
			CppCtorDecl decl(SStr(name, pos), []);
			// 'decl' will not create the constructor without a body.
			add(CppCtor(cppScope(this), SrcPos(), this, decl));
		}

		if (!hasCopyCtor) {
			add(TypeCopyCtor(this));
		}

		if (!hasAssignFn) {
			add(TypeAssign(this));
		}

		// Add a destructor if we don't have one.
		if (!hasDtor) {
			CppDtor dtor(cppScope(this), SrcPos(), this, null);
			add(dtor);
			add(wrapCppDtor(dtor));
		}

		super:loadAll();
	}
}

// Is this a copy-ctor or assign fn?
private Bool isCopyFn(Type type, Named fn) {
	if (fn.params.count == 2) {
		// Storm-style.
		return fn.params[1].type is type;
	}

	if (fn.params.count == 3) {
		// C++ style
		if (fn.params[1].type !is named{Nat})
			return false;

		if (ref = fn.params[2].type as PtrType) {
			return ref.isRef & ref.inside is type;
		}
		return false;
	}

	false;
}

/**
 * Member variables with support for 'const'.
 */
class CppMemberVar extends MemberVar {
	init(SrcPos pos, Str name, CppValue type, Type memberOf) {
		init(pos, name, type, memberOf) { const = type.const; }
	}

	Bool const;
}

/**
 * Member variable access.
 */
class MemberVarAccess extends Expr {
	MemberVar var;
	Expr in;

	init(SrcPos pos, Expr in, MemberVar var) {
		init(pos) { in = in; var = var; }
	}

	Bool temporary() : override {
		false;
	}

	Bool constant() : override {
		if (var as CppMemberVar)
			return var.const;
		false;
	}

	ExprResult result() : override {
		var.type.asRef();
	}

	void code(CodeGen gen, CodeResult result) : override {
		if (!result.needed())
			return;

		CodeResult lhs(in.result.type.asRef(), gen.block);
		in.code(gen, lhs);

		if (result.type.ref) {
			Var r = result.location(gen);
			gen.l << mov(r, lhs.location(gen));
			gen.l << add(r, ptrConst(var.offset));
			result.created(gen);
		} else {
			var r = result.location(gen);
			gen.l << mov(ptrA, lhs.location(gen));
			gen.l << add(ptrA, ptrConst(var.offset));
			if (result.type.isAsmType) {
				gen.l << mov(r, xRel(r.size, ptrA));
			} else {
				gen.l << lea(ptrB, r);
				gen.l << fnParam(ptrDesc, ptrB);
				gen.l << fnParam(ptrDesc, ptrA);
				gen.l << fnCall(result.type.copyCtor, true);
			}
		}
	}

	void codePtr(CodeGen gen, Type type) : override {
		in.codePtr(gen, type);
		gen.l << add(ptrC, ptrConst(var.offset));
	}
}
